/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.db;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Vector;

import edu.udo.cs.miningmart.compiler.CompilerAccessLogic;
import edu.udo.cs.miningmart.exception.DbConnectionClosed;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.M4InterfaceContext;
import edu.udo.cs.miningmart.m4.RelationalDatatypes;


/**
 * This class provides an access to a MySQL DBMS.
 * It has only been tested with MySQL version 5.0.3.
 * See www.mysql.com
 * 
 * To compile this class, the MySQL JDBC driver package must
 * be added to the classpath.
 * 
 * @author Timm Euler
 * @version $Id: DbCoreMysql.java,v 1.4 2006/10/02 08:58:56 euler Exp $
 */
public class DbCoreMysql extends DbCore {
	
	/** The public constant indicating the Mysql Datatype DOUBLE */
	public static final String MYSQL_TYPE_NUMBER = "DOUBLE";

	/** The public constant indicating the Mysql Datatype INTEGER */
	public static final String MYSQL_TYPE_INTEGER = "INTEGER";
	
	/** The public constant indicating the Mysql Datatype VARCHAR */
	public static final String MYSQL_TYPE_STRING = "VARCHAR";
	
	/** The public constant indicating the Mysql Datatype CHAR */
	public static final String MYSQL_TYPE_CHAR = "CHAR";
	
	/** The public constant indicating the Mysql Datatype DATE */
	public static final String MYSQL_TYPE_DATE = "DATE";
	
	/** The public constant indicating the Mysql Datatype TIME */
	public static final String MYSQL_TYPE_TIME = "TIME";
	
	/** The public constant indicating the Mysql Datatype TIMESTAMP */
	public static final String MYSQL_TYPE_TIMESTAMP = "DATETIME";
	
	/** The public constant indicating the Mysql Datatype TEXT */
	public static final String MYSQL_TYPE_TEXT = "TEXT";
	
    /**
     * The constructor of this class calls the constructor of the
     * superclass DbCore.
     * Autocommit is set to false, so transaction management must be done
     * individually by the database functions.
     *
     * @param url Prefix of the database url
     * @param dbName name of the database
     * @param user name of the database user
     * @param passwd password for the database user
     * @param cal the <code>CompilerAccessLogic</code> of the current thread. It is just needed
     * 		  in order to reach the appropriate <code>Print</code>-object for writing debug messages
     * @param isM4Schema is <code>true</code> iff the connection refers to the M4 schema. Useful
     * 		  to decide whether triggers need to be switched off etc.
     * 
     * @see edu.udo.cs.miningmart.m4.core.utils.DbCore#DbCore(String, String, String, String, CompilerAccessLogic, boolean)
     * @throws SQLException
     */
	public DbCoreMysql( String url,
						String dbName,
						String user,
						String passwd,
		                M4InterfaceContext cal,
		                boolean isM4Schema)
	throws SQLException	{
		super( url,
			   "/" + dbName,
			   user,
			   passwd,
			   cal,
			   isM4Schema);
		// under Mysql we use the database name as schema:
    	this.mySchemaName = dbName;
	}

	/**
	 * Implements the superclass method so that the correct
	 * JDBC driver for MySQL is used.
	 */
	protected void registerJDBC_Driver() throws SQLException {
		DriverManager.registerDriver(new com.mysql.jdbc.Driver());
	}

	String jdbcDriverClassName() {
		return "com.mysql.jdbc.Driver";
	}
	
	String getDateComparisonAsSqlString(String oneDate, String secondDate, boolean greater, boolean orEqual, String timeFormat) {
		// timeFormat can be ignored as DATEDIFF extracts the date itself
		String ret = 
			"DATEDIFF(" + oneDate + ", '" + secondDate + "') " +
			this.getComparisonOp(greater, orEqual) + " 0";
		return ret;
	}
	
	/** 
	 * Indicates the DBMS used. 
	 * 
	 * @return The public static constant DB.MYSQL
	 */
	public short getDbms() {
		return DB.MYSQL;
	}
	
	String installDir() {
		return "mysql";
	}
	
	/**
	 * @see DbCore#switchAutocommitOff()
	 */
	protected void switchAutocommitOff(Connection con) throws SQLException {
		con.setAutoCommit(false);
	}
	
	/**
 	 * @see DbCore#getNextM4SequenceValue(Statement)
	 */
	protected long getNextM4SequenceValue(Statement stmt)
	throws M4Exception {
		String tableWithM4Ids = "m4sequence_t";
		// in mysql a different mechanism is used because no
		// database sequences are supported. so the highest 
		// global m4 id is always stored in a specific table.
		String query = "select m4_id from " + tableWithM4Ids;
		Long currentval = null;
		long nextval = -1;
	    try {	    	
			currentval = this.executeSingleValueSqlReadL(query, stmt);
			if (currentval != null) {
				query = "delete from " + tableWithM4Ids;
				this.executeSqlWrite(query, stmt);
				nextval = currentval.longValue() + 1;
				query = "insert into " + tableWithM4Ids + " values (" + nextval + ")";
				this.executeSqlWrite(query, stmt);
				commitBatch(stmt);
				return nextval;
			}
	    }
	    catch (SQLException e) {
	    	throw new M4Exception("An SQL error occurred when reading next M4 sequence value: " + e.getMessage());
	    }
	    throw new M4Exception("Could not determine next M4 sequence value!");
	}
	
	long getNextM4SequenceValue() throws DbConnectionClosed, M4Exception {

		try { 
			Statement stmt = this.getDbWriteStatement();
			return this.getNextM4SequenceValue(stmt);
		}
		catch (SQLException e) { // Only possible origin is "this.getM4DbStatement()"!
			String msg = "Error in method DbCore.getNextM4SequenceValue():\n"
					   + "Could not create Statement:\n"
					   + e.getMessage();
			throw new M4Exception(msg);			
		}
	}
	
	String getTableOwner() {
		return this.mySchemaName;
	}
	
	/**
	 * @see DbCore
	 * 
	 * @return An SQL String
	 */
	public String getSelectStringAllTables() {
		String ret = "select table_name from information_schema.tables where table_schema = '"
						+ this.getTableOwner() + "' and table_type = 'BASE TABLE'";

		return ret;		
	}

	/**
	 * @see DbCore
	 * 
	 * @return An SQL String
	 */
	public String getSelectStringAllViews() {
		String ret = "select table_name from information_schema.tables where table_schema = '"
			+ this.getTableOwner() + "' and table_type = 'View'";

		return ret;		
	}

	/**
	 * @see DbCore
	 */
	public String getSelectStringDistribValuesTimeColumn(
			String colSql, String tableName, String distribBase) {

		if (colSql == null || tableName == null || distribBase == null)
			return null;
		String formatToExtract = null;
		if (distribBase.equalsIgnoreCase("YYYY")) {
			formatToExtract = "YEAR";
		}
		if (distribBase.equalsIgnoreCase("YYYYQ")) {
			formatToExtract = "QUARTER"; // output format is not quite like in postgres or oracle
		}
		if (distribBase.equalsIgnoreCase("YYYYMM")) {
			formatToExtract = "YEAR_MONTH";
		}
		if (distribBase.equalsIgnoreCase("YYYYMMDD")) {
			formatToExtract = "DATE";
		}
		if (formatToExtract == null)
			return null;
		
		String columnQuery = "EXTRACT (" + formatToExtract + " FROM " + colSql + ")";
		if (formatToExtract.equalsIgnoreCase("DATE")) {
			columnQuery = "DATE(" + colSql + ")";
		}
		
		String query = "SELECT " +
		               columnQuery + ", " +
					   "COUNT(*)," +
					   "MIN(" + columnQuery + "), " +
					   "MAX(" + columnQuery + ") " +
					   "FROM " + tableName +
					   " WHERE (" + columnQuery + ") IS NOT NULL "+
					   "GROUP BY " + columnQuery +
					   " ORDER BY " + columnQuery;
		
		/* Oracle/Postgres version:
		 *     "SELECT " 
			 + " TO_CHAR(" + colSql + ", '" + distribBase + "'),"
    		 + " COUNT(*),"
    		 + " TO_CHAR(MIN(" + colSql + "), '" + distribBase + "'),"
    		 + " TO_CHAR(MAX(" + colSql + "), '" + distribBase + "')"
    		 + " FROM " + tableName
			 + " WHERE (" + colSql + ") IS NOT NULL"
			 + " GROUP BY TO_CHAR(" + colSql + ", '" + distribBase + "')"
			 + " ORDER BY TO_CHAR(" + colSql + ", '" + distribBase + "')";
		*/
		
		return query;
	}
	
	/**
	 * @see DbCore
	 * 
	 * @param dbObjectName Name of a table OR view in the business data schema
	 * @return An SQL String
	 */
	public String getSelectStringAllColumnsForDbObject(String dbObjectName) {
	    return "SELECT column_name, column_type " +
	           "FROM information_schema.columns " +
	           "WHERE table_name = '" + dbObjectName.toLowerCase() + "'";
	}

	public long getNumberOfMonthsBetween(String dbObjectName, String firstValue, String secondValue) 
	throws M4Exception {
		try {
			String yearQuery = "SELECT EXTRACT(YEAR_MONTH FROM " + firstValue + ") FROM " + dbObjectName;
			Long firstYear = this.executeSingleValueSqlReadL(yearQuery);
			yearQuery = "SELECT EXTRACT(YEAR_MONTH FROM " + secondValue + ") FROM " + dbObjectName;
			Long secondYear = this.executeSingleValueSqlReadL(yearQuery);
			if (firstYear == null || secondYear == null) {
				return 0;
			}
			long monthDiff = firstYear.longValue() - secondYear.longValue();
			if (monthDiff < 0) monthDiff *= -1;

			return monthDiff;
		}
		catch (SQLException s) {
			throw new M4Exception("Sql error trying to compute number of months between '" +
					firstValue + "' and '" + secondValue + "': " + s.getMessage());
		}
		catch (DbConnectionClosed d) {
			throw new M4Exception("No connection to DB when trying to compute number of months between '" +
					firstValue + "' and '" + secondValue + "': " + d.getMessage());
		}
	}
	
	public boolean tableExists(String tableName) throws M4Exception {
		String query = "SELECT table_name FROM information_schema.tables WHERE table_name = '" +
						tableName.toLowerCase() + "'";
		if (this.getTableOwner() != null) {
			query += " AND table_schema = '" + this.getTableOwner() + "'";
		}
		String l = null;
		try {
			this.commitTransactions();
			l = this.executeSingleValueSqlRead(query);
		}
		catch (SQLException sqle) {
			throw new M4Exception("SQL error testing if table '" + tableName + "' exists: " + sqle.getMessage());
		}
		catch (DbConnectionClosed d) {
			throw new M4Exception("DB connection error when testing if table '" + tableName + "' exists: " + d.getMessage());
		}
		return (l != null);
	}
	
	/**
	 * @see DbCore
	 * 
	 * @param dbObjectName Name of a table or view in the business data schema
	 * @param owner Name of the owner of the table or view (can be null)
	 * @param columnName Name of the column whose datatype is returned
	 * @return the DBMS-dependent name of the datatype of the column with the given name
	 */
	public String getSelectStringColumnDataTypes(String dbObjectName, String owner, String columnName) {
	    String ret = "SELECT column_type " +
    		   "FROM information_schema.columns " +
	           "WHERE table_name = '" + dbObjectName.toLowerCase() + 
	           "' AND column_name = '" + columnName.toLowerCase() + "'";
	    if (owner != null) {
	    	// in mysql there are no owners, only databases:
	    	if ( ! owner.toLowerCase().equals(this.getTableOwner()))
	    		owner = this.getTableOwner();
	    	ret += " AND table_schema = '" + owner.toLowerCase() + "'";
	    }
	    return ret;
	}

	/**
	 * @see DbCore
	 */
	public Collection getPrimaryKeyColumnNames(String dbObjectName) 
	throws SQLException, DbConnectionClosed {
		Vector thePrimaryKeyColumnNames = new Vector();
		
		String query = "SELECT column_name FROM " +
						"information_schema.key_column_usage WHERE " +
						"table_name = '" + dbObjectName.toLowerCase() +
						"' and constraint_name = 'PRIMARY'";
		ResultSet rs = this.executeSqlRead(query);
		while (rs.next()) {
			thePrimaryKeyColumnNames.add(rs.getString(1));
		}
		rs.close();
		if (thePrimaryKeyColumnNames.isEmpty())
			return null;
		return thePrimaryKeyColumnNames;
	}
	
	/**
	 * @see DbCore
	 */
	public Map getTablesReferencedBy(String dbObjectName) 
	throws SQLException, DbConnectionClosed {
		Map myMap = new HashMap();
		String query = "SELECT column_name, referenced_table_name " +
						"FROM information_schema.key_column_usage " +
						"WHERE table_name = '" + 
						dbObjectName.toLowerCase() + 
						"' and table_schema = '" + this.getTableOwner() + "'";
		ResultSet rs = this.executeSqlRead(query);
		while (rs.next()) {
			myMap.put(rs.getString(1), rs.getString(2));
		}
		rs.close();
		return myMap;
	}
	
	/**
	 * @see DbCore
	 */
	public Map getTablesReferringTo(String dbObjectName) 
	throws SQLException, DbConnectionClosed {
		Map myMap = new HashMap();
		String query = "SELECT column_name, table_name " +
						"FROM information_schema.key_column_usage " +
						"WHERE referenced_table_name = '" + 
						dbObjectName.toLowerCase() + 
						"' and referenced_table_schema = '" + this.getTableOwner() + "'";
		ResultSet rs = this.executeSqlRead(query);
		while (rs.next()) {
			String colName = rs.getString(1);
			String tableName = rs.getString(2);
			Collection listOfTables = (Collection) myMap.get(colName);
			if (listOfTables == null) {
				listOfTables = new Vector();
				myMap.put(colName, listOfTables);
			}
			if ( ! listOfTables.contains(tableName))
				listOfTables.add(tableName);
		}
		rs.close();
		return myMap;
	}

	/**
	 * Returns one of the TYPE_... constants in the class
	 * edu.udo.cs.miningmart.m4.Columnset, or null if the given 
	 * String is neither a table or view.
	 * 
	 * @param dbObjectName name of a table or view
	 * @return a String with the type information
	 */
	public String getTableOrViewType(String dbObjectName) 
	throws SQLException, DbConnectionClosed {
		String query = "SELECT table_type FROM information_schema.tables WHERE table_name = '" +
				dbObjectName.toLowerCase() + "'";
		String type = this.executeSingleValueSqlRead(query);
		if (type == null)
			return null;
		if (type.equalsIgnoreCase("VIEW")) {
			return Columnset.TYPE_VIEW;
		}
		if (type.equalsIgnoreCase("BASE TABLE")) {
			return Columnset.TYPE_TABLE;
		}
		return null;
	}
	
	/**
	 * @see DbCore
	 * 
	 * @return the name of the column that contains the column names
	 */
	public String getAttributeForColumnNames() {
	    return "column_name";
	}
	
	/**
	 * @see DbCore
	 * 
	 * @return the name of the column that contains the column types
	 */
	public String getAttributeForColumnTypes() {
	    return "column_type";
	}
	
	/**
	 * WARNING: the unique row identifier works only for tables 
	 * with a single primary key column of type integer!
	 * @see DbCore
	 */
	public String getUniqueRowIdentifier() {
		// no such thing as ROWNUM under Mysql
		return null;
	}

	/**
	 * @see DbCore
	 */
	public String getPowerExpression(String base, String exponent) {
		return "POWER(" + base + ", " + exponent + ")";
	}
	
	/**
	 * {@inheritDoc}
	 */
	public String getFloorExpression() {
		return "floor";
	}
	
	/**
	 * {@inheritDoc}
	 */
	public String getRoundToDecimalPlacesExpression() {
		return "truncate";
	}
	
	/**
	 * @see DbCore
	 */
	public String getAliasExpressionForInnerSelects(String aliasName) {
		return " AS " + aliasName;
	}
	
	/**
	 * @see DbCore
	 */
	public String getDatatypeName(String m4RelDatatypeName, int size) {
		String ret = null;
		if (m4RelDatatypeName.equals(RelationalDatatypes.RELATIONAL_DATATYPE_NUMBER)) {
			ret = MYSQL_TYPE_NUMBER;
		}
		else if (m4RelDatatypeName.equals(RelationalDatatypes.RELATIONAL_DATATYPE_DATE)) {
			ret = MYSQL_TYPE_DATE;
		}
		else if (m4RelDatatypeName.equals(RelationalDatatypes.RELATIONAL_DATATYPE_KEY)) {
			ret = MYSQL_TYPE_INTEGER; //??
		}
		else if (m4RelDatatypeName.equals(RelationalDatatypes.RELATIONAL_DATATYPE_STRING)) {
			ret = MYSQL_TYPE_STRING;
		}
		
		if ((ret != null) && (size > 0))
		{   ret += "(" + size + ")";   }
		else
			if ((ret != null) && ret.equals(MYSQL_TYPE_STRING))
				ret += "(100)"; // VARCHAR is not allowed in Mysql, VARCHAR(xx) is
		
		return ret;
	}		
	
	/**
	 * @see DbCore
	 */
	public String getM4DatatypeName(String dbmsDatatypeName) {
		dbmsDatatypeName = dbmsDatatypeName.toUpperCase();
		if (dbmsDatatypeName.equals(MYSQL_TYPE_NUMBER)) {
			return RelationalDatatypes.RELATIONAL_DATATYPE_NUMBER;
		}
		else if (dbmsDatatypeName.startsWith("FLOAT")) {
			return RelationalDatatypes.RELATIONAL_DATATYPE_NUMBER;
		}
		else if (dbmsDatatypeName.startsWith("INT")) {
			return RelationalDatatypes.RELATIONAL_DATATYPE_NUMBER;
		}
		else if (dbmsDatatypeName.startsWith("BIGINT")) {
			return RelationalDatatypes.RELATIONAL_DATATYPE_NUMBER;
		}
		else if (dbmsDatatypeName.startsWith("SMALLINT")) {
			return RelationalDatatypes.RELATIONAL_DATATYPE_NUMBER;
		}
		else if (dbmsDatatypeName.startsWith("TINYINT")) {
			return RelationalDatatypes.RELATIONAL_DATATYPE_NUMBER;
		}
		else if (dbmsDatatypeName.startsWith("DECIMAL")) {
			return RelationalDatatypes.RELATIONAL_DATATYPE_NUMBER;
		}
		else if (dbmsDatatypeName.equals(MYSQL_TYPE_DATE) ||
				 dbmsDatatypeName.equals(MYSQL_TYPE_TIME) ||
				 dbmsDatatypeName.equals(MYSQL_TYPE_TIMESTAMP)) {
			return RelationalDatatypes.RELATIONAL_DATATYPE_DATE;
		}
		else if (dbmsDatatypeName.startsWith(MYSQL_TYPE_STRING)) {
			return RelationalDatatypes.RELATIONAL_DATATYPE_STRING;
		}
		else if (dbmsDatatypeName.equals(MYSQL_TYPE_TEXT)) {
			return RelationalDatatypes.RELATIONAL_DATATYPE_STRING;
		}
		return null;
	}
	
	/**
	 * Try to drop the table or view with the given name, if it exists. Returns TRUE iff the table
	 * had existed, FALSE otherwise.
	 * 
	 * @param tableName the name of the table to be dropped
	 * @return TRUE iff the table had existed before it was removed,
	 *         FALSE if nothing was done.
	 */
	public boolean dropRelation(String tableName) throws M4Exception {
		
		boolean tableExists = this.tableExists(tableName);
		
		if (tableExists) {
			String sql_drop = "DROP TABLE " + tableName;
				
			try
			{	this.executeSqlWrite(sql_drop);  }
			catch (SQLException sqle) {
				throw new M4Exception(
					"Error trying to remove the table '"
					+ tableName + "':\n" + sqle.getMessage());
			}
			catch (DbConnectionClosed dbe) {
				throw new M4Exception(
					"DB Connection closed when deleting table '"
					+ tableName + "':\n" + dbe.getMessage());
			}    
		}		          
		               
		return tableExists;
	}
	
	/** @return a <code>String</code> that can be used as a query to test the DB connection. */
	public String getTestQuery() {
		return getSelectStringAllTables();	
	}
	
	/**
	 * Creates a primary key constraint in the database. 
	 * 
	 * @param tableName The table for which to add the constraint
	 * @param pkAttributeNames a <code>Collection</code> of <code>String</code>s, each
	 *        with the name of a primary key attribute (relational level)  
	 * @param dbConstraintName a unique name for the database constraint,
	 *        or <code>null</code> to use a generic name 
	 * @return the name of the database constraint 
	 */
	public String createPrimaryKeyConstraint(String tableName, Collection pkAttributeNames, String dbConstraintName)
		throws SQLException, DbConnectionClosed
	{
		if (dbConstraintName == null || dbConstraintName.trim().length() == 0) {
			dbConstraintName = tableName + "_PK";
		}
		
		String commaSeparatedKeyNames = stringCollectionToCommaSeparated(pkAttributeNames);
		
		String sql = "ALTER TABLE " + tableName
			       + " ADD CONSTRAINT " + dbConstraintName
				   + " PRIMARY KEY (" + commaSeparatedKeyNames + ")";
		
		this.executeSqlWrite(sql);
		return dbConstraintName;
	}

	/**
	 * This method creates a foreign key constraint in the database. 
	 * 
	 * @param fkTableName The name of the table to create the constraint for.
	 * @param fkAttributeNames <code>Collection</code> of attribute names (relational level)
	 *    part of the <code>fkTableName</code> table that point to the primary key attributes
	 *    of <code>pkTableName</code>.
	 * @param pkTableName The name of the referenced table.
	 * @param pkAttributeNames <code>Collection</code> of attribute names that constitute the
	 *    primary key of the referenced table
	 * @param dbConstraintName unique name of the database constraint, must not be
	 *    <code>null</code> 
	 * @throws SQLException
	 * @throws DbConnectionClosed
	 */
	public void createForeignKeyConstraint(String fkTableName, Collection fkAttributeNames,
			String pkTableName, Collection pkAttributeNames, String dbConstraintName)
	throws SQLException, DbConnectionClosed
	{
		String fkNamesComma = stringCollectionToCommaSeparated(fkAttributeNames);
		String pkNamesComma = stringCollectionToCommaSeparated(pkAttributeNames);
		
		
		String sql = "ALTER TABLE " + fkTableName
		           + " ADD CONSTRAINT " + dbConstraintName
				   + " FOREIGN KEY (" + fkNamesComma
				   + ") REFERENCES " + pkTableName
				   + "(" + pkNamesComma + ")";
	
		this.executeSqlWrite(sql);
	}

}
/*
 * Historie
 * --------
 *
 * $Log: DbCoreMysql.java,v $
 * Revision 1.4  2006/10/02 08:58:56  euler
 * Code repairs
 *
 * Revision 1.3  2006/10/01 19:14:22  euler
 * Mysql works
 *
 * Revision 1.2  2006/09/30 14:20:19  euler
 * some fixes, still buggy with mysql
 *
 * Revision 1.1  2006/09/29 17:19:26  euler
 * Still some mysql bugs
 *
 * Revision 1.17  2006/09/27 15:00:02  euler
 * New version 1.1
 *
 * Revision 1.16  2006/09/02 12:59:33  euler
 * *** empty log message ***
 *
 * Revision 1.15  2006/08/21 13:59:07  euler
 * Bugfixes
 *
 * Revision 1.14  2006/08/10 14:38:02  euler
 * New mechanism for reversing steps
 *
 * Revision 1.13  2006/06/18 15:13:06  euler
 * Bugfixes
 *
 * Revision 1.12  2006/06/16 17:30:41  scholz
 * update
 *
 * Revision 1.11  2006/06/14 14:26:20  scholz
 * cleaned up code
 *
 * Revision 1.10  2006/06/14 13:28:52  scholz
 * moved creation of integrity constraints to DbCore
 *
 * Revision 1.9  2006/05/22 20:11:52  euler
 * *** empty log message ***
 *
 * Revision 1.8  2006/04/11 14:10:16  euler
 * Updated license text.
 *
 * Revision 1.7  2006/04/06 16:31:15  euler
 * Prepended license remark.
 *
 * Revision 1.6  2006/03/29 11:16:44  euler
 * Still more robust when installing.
 *
 * Revision 1.5  2006/03/29 09:50:47  euler
 * Added installation robustness.
 *
 * Revision 1.4  2006/03/04 12:13:44  euler
 * *** empty log message ***
 *
 * Revision 1.3  2006/01/24 12:31:45  euler
 * Added recognition of key type for new columns.
 * Removed EstimatedStatistics from context menu
 * because they are in too basic status for the release.
 *
 * Revision 1.2  2006/01/03 15:21:44  euler
 * Bugfixes
 *
 * Revision 1.1  2006/01/03 09:54:23  hakenjos
 * Initial version!
 *
 */
