/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.storedProcedures;
import java.sql.SQLException;
import java.util.Vector;

/** 
 * This class realizes the M4 operator &quot;Windowing&quot;.
 * @author Martin Scholz
 * @version $Id: ToWindow.java,v 1.5 2006/04/11 14:10:16 euler Exp $
 */
public class ToWindow {

    private final DbConnector connector;
    private final Windowing  window;   // for the input
    private final OutputColumnset out; // for the output
    private final int windowSize;

    protected Windowing getWindow() {
	return this.window;
    }

    protected OutputColumnset getOutputColumnset() {
	return this.out;
    }

    protected int getWindowSize() {
	return this.windowSize;
    }

    /**
     * Constructor
     * @param dbc source of database connection if not used as stored procedure
     * @param source source table
     * @param time   time column of source table
     * @param column value column of source table
     * @param target name of the target table to be created
     * @param outTimeStart name of the interval start attribute for
     * the output table
     * @param outTimeEnd name of the interval end attribute for the
     * output table
     * @param targetColumns String array containing the name of the
     * windowed output columns. Specifies the window size!
     * @throws SQLException
     * @throws TimeOperatorException
    */
    public ToWindow(BusinessDbConnectionSource dbc,
    				String source, String time, String column,
					String target, String outTimeStart, String outTimeEnd,
					String[] targetColumns, int distance)
		throws SQLException, TimeOperatorException
	{
		this.connector = new DbConnector(dbc);
		this.windowSize = targetColumns.length;
		final String timeDT =
			DataType.getColumnDataType(source, time, this.connector);
		final String columnDT =
			DataType.getColumnDataType(source, column, this.connector);

		// Input:
		final Attribute timeAttr = new Attribute(time, timeDT, timeDT);
		final Attribute columnAttr =
			new Attribute(column, columnDT, dbc.getNameOfNumericDatatype());
		if (!timeAttr.isConvertable())
			DataType.wrongDataType(target, timeAttr);
		if (!columnAttr.isConvertable())
			DataType.wrongDataType(target, columnAttr);

		// Output:
		final Attribute timeStart = new Attribute(outTimeStart, timeDT, timeDT);
		final Attribute timeEnd = new Attribute(outTimeEnd, timeDT, timeDT);
		final Attribute[] allAttributes =
			new Attribute[targetColumns.length + 2];
		allAttributes[0] = timeStart;
		allAttributes[1] = timeEnd;
		for (int i = 0; i < targetColumns.length; i++)
			allAttributes[i + 2] =
				new Attribute(targetColumns[i], columnDT, columnDT);
		this.out =
			new OutputColumnset(
				target,
				allAttributes,
				this.connector.getConnection());

		// Prepare Windowing:
		this.window =
			new Windowing(targetColumns.length, distance, this.connector);
		if (!window
			.open(
				"SELECT "
					+ columnAttr.getConvertedInput()
					+ ", "
					+ timeAttr.getConvertedInput()
					+ " FROM "
					+ source)) {
			DbConnector.infoOutput("Error while opening " + source);
			throw new TimeOperatorException(
				"Could not read from table '" + source + "'!");
		}
	}

    /**
     * Method calc() reads from the column(s) specified in the constructor
     * and writes to the specified target, using windowing.
     * <!-- Finally an index is created on the target table. -->
     */
    public void calc() throws TimeOperatorException {
       	String[] ret = null;
		do {
		    ret = getWindow().getNextWnd();
	    	String start = window.getTimeFirstEntry();
		    String end   = window.getTimeLastEntry();
		    insert(start, end, ret);
		} while (ret != null);

		// this.getOutputColumnset().createIndexOn(INDEXONATTRIB);
		// *** No index is created in this compiler version! ***
	
		this.getWindow().close();
		this.connector.close();
    }


    /* description: inserts a window of values into the target-table
     * @param start point int time where the interval starts
     * @param end   point int time where the interval ends
     * @param value vector of values to be entered 
    */
    private void insert(String start, String end, String[] value)
	throws TimeOperatorException
    {
	if (value == null)
	    return;

	final String[] allValues = new String[this.getWindowSize()+2];
	allValues[0] = start;
	allValues[1] = end;
	for (int i=0; i<value.length; i++)
	    allValues[i+2] = value[i];

	this.getOutputColumnset().insert(allValues);
    }

    
    /** 
     * --- dbWindow ---
     * description : this function has to be used in the database as stored procedure
     *		   for rerepresenting a time series using windows.
     *		   the datatypes will be mapped to SQL datatypes.
     * in : see the constructor for more information
     * targetColumns: space separated list of target column names
     */
    public static void dbWindow(Object dbc,
    			String sourceTable, String timeAttrib, String valueAttrib,
				String targetTable, String outTimeStart, String outTimeEnd,
				String outWindowedCols, int distance)
		throws SQLException, TimeOperatorException
	{
		String[] tcArray = parseNamesList(outWindowedCols);
		BusinessDbConnectionSource dbcCasted =
			(dbc instanceof BusinessDbConnectionSource) ?
				((BusinessDbConnectionSource) dbc) : null;
		
		ToWindow tw =
			new ToWindow(dbcCasted, sourceTable, timeAttrib, valueAttrib,
						 targetTable, outTimeStart, outTimeEnd,
						 tcArray, distance);
		tw.calc();
		tw = null;
	}
	
    /** private helper method */
    private static String[] parseNamesList (String s) {
		if (s == null) {
	    	return null;
		}
				
		Vector values = new Vector();
		int stop = 0;
		for (int from = 0; from < s.length(); from = stop) {
			from = nextNonWhitespace(s, from);
			if (from == s.length())
				break;
			stop = nextWhitespace(s, from);
			String sub = s.substring(from, stop);
			values.add(sub);
		}
		String names[] = new String[values.size()];
		for (int arg = 0; arg < names.length; arg++)
			names[arg] = (String) values.elementAt(arg);
		return names;
	}

    /** private helper method */
    private static int nextNonWhitespace(String s, int index) {
		for (;((index<s.length())&&(Character.isWhitespace(s.charAt(index))));index++);
		return (index);
    }

    /** private helper method */
    private static int nextWhitespace(String s, int index) {
		for (;((index<s.length())&&(!Character.isWhitespace(s.charAt(index))));index++);
		return (index);
    }

}
/*
 *$Log: ToWindow.java,v $
 *Revision 1.5  2006/04/11 14:10:16  euler
 *Updated license text.
 *
 *Revision 1.4  2006/04/06 16:31:15  euler
 *Prepended license remark.
 *
 *Revision 1.3  2006/03/30 16:07:15  scholz
 *fixed author tags for release
 *
 *Revision 1.2  2006/03/29 09:50:47  euler
 *Added installation robustness.
 *
 *Revision 1.1  2006/01/03 09:54:23  hakenjos
 *Initial version!
 * 
 */
