/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;

import java.util.Collection;
import java.util.Iterator;

import edu.udo.cs.miningmart.compiler.utils.RandomPartition;

import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.Column;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Value;

/**
 * This operator segments an input concept randomly.
 * 
 * @see edu.udo.cs.miningmart.m4.core.operator.Segmentation
 * 
 * @author Martin Scholz
 * @version $Id: SegmentationByPartitioning.java,v 1.4 2006/04/11 14:10:11 euler Exp $
 */
public final class SegmentationByPartitioning extends Segmentation {

	private static final String MULTISTEPBRANCH_ATTRIBUTE = "(Random)";

	/**
	 * Overrides the method from Segmentation.java because a nested view definition is needed.
	 * 
	 * @see edu.udo.cs.miningmart.m4.core.operator.ConceptOperator#generateSQLDefinition(String, int)
	 */
    public String generateSQLDefinition(String selectPart, int index) throws M4CompilerError
    {
		final int howMany    = new Integer(getHowManyPartitions().getValue()).intValue();
		final Columnset cs;
		try {
			cs = this.getInputConcept().getCurrentColumnSet();
		}		
   		catch (M4Exception m4e)
   		{   throw new M4CompilerError("M4 interface error in " + this.getName() + ": " + m4e.getMessage());  } 
		
		String tempTableName = this.getTempCsName(cs);
		
		if (index <= 0) { // Create the temporary table with the row numbers per segment just once!
			new RandomPartition( cs,
								 null,
								 new String[howMany],
								 true,
								 tempTableName,
								 null,
								 null,
								 this.getM4Db()
			);
			this.getM4Db().addTableToTrash(tempTableName, cs.getSchema(), this.getStep().getId());
		}    

    	String artCol = this.getArtificalColumnName(selectPart);
		String sqlDefs = this.getSourceAttributeDefinitions();
		String colNames  = this.getSourceAttributes();
    
    	// Construct query that joins the tables:
		String innerQuery =
				"( SELECT ROWNUM AS " + artCol + ", "
				+ sqlDefs + " FROM "
				+ cs.getSchemaPlusName() + " )";
		
		String viewDef =
					"(SELECT " + colNames
					+ " FROM " + tempTableName + " T, "
					+ innerQuery + " S"
					+" WHERE t.the_Segment = " + index
					+ " AND s." + artCol + " = t.the_Id)";
     
		return viewDef;
		    
    } // end public Columnset generateColumnsetForOP

	/**
	 * @see edu.udo.cs.miningmart.m4.core.utils.RandomPartition#getSourceAttributeDefinitions()
	 * */
	public String getSourceAttributeDefinitions() throws M4CompilerError
	{
		try {
			Columnset cs = this.getInputConcept().getCurrentColumnSet();
			Collection columns = cs.getColumns();
			StringBuffer sb  = new StringBuffer();
			Iterator it = columns.iterator();
		 	while (it.hasNext()) {
				Column theColumn = (Column) it.next();
				String sql  = theColumn.getSQLDefinition();
				String name = theColumn.getName();
				if (sql == null || sql.equals(name)) {
					sb.append(name);
				}
				else {
					sb.append("(" + sql + ") AS " + name);
				}
				sb.append(", ");
			}
			int le = sb.length();
			if (le >= 2) {
				sb.delete(le - 2, le);
			}
			return sb.toString();
		}
   		catch (M4Exception m4e)
   		{   throw new M4CompilerError("M4 interface error in " + this.getName() + ": " + m4e.getMessage());  } 
	}

	/**
	 * @see edu.udo.cs.miningmart.m4.core.utils.RandomPartition#getSourceAttributes()
	 * */
	public String getSourceAttributes() throws M4CompilerError
	{
		try {
			Columnset cs = this.getInputConcept().getCurrentColumnSet();
			Collection columns = cs.getColumns();
			StringBuffer sb  = new StringBuffer();
			Iterator it = columns.iterator();
		 	while (it.hasNext()) {
				String colName = ((Column) it.next()).getName().toUpperCase();
				sb.append(colName + ", ");
			}
			int le = sb.length();
			if (le >= 2) {
				sb.delete(le - 2, le);
			}
			return sb.toString();
		}
   		catch (M4Exception m4e)
   		{   throw new M4CompilerError("M4 interface error in " + this.getName() + ": " + m4e.getMessage());  } 
	}


	/**
	 * We need to take care, that we always reference the same temporary table, as long as we
	 * work on the same source columnset!
	 * */
    String getTempCsName(Columnset inputCs) {
    	return "T_" + this.getStep().getId() + "_" + inputCs.getId();
    }

	/**
	 * @see edu.udo.cs.miningmart.m4.core.operator.ConceptOperator#getTypeOfNewColumnSet(int)
	 */
    public String getTypeOfNewColumnSet(int index)
    {
        // all generated Colunsets are views
        return "V";
    }
    
	/**
	 * Only for the java compiler.
	 * 
	 * @see edu.udo.cs.miningmart.m4.core.operator.Segmentation#generateConditionForOp(int)
	 */
    public String generateConditionForOp(int columnSetIndex)
    	throws M4CompilerError
    {
		return null;
    }

	/**
	 * Getter method for parameter "HowManyPartitions".
	 * 
	 * @return the parameter "HowManyPartitions" as a Value
	 */
    public Value getHowManyPartitions() throws M4CompilerError {
    	return (Value) this.getSingleParameter("HowManyPartitions");
    }

	/**
	 * @see edu.udo.cs.miningmart.m4.core.operator.Segmentation#numberOfSegments
	 */
    public int numberOfSegments() throws M4CompilerError
    {
    	try {
    		return Integer.parseInt(this.getHowManyPartitions().getValue());
    	}
        catch (NumberFormatException nfe) {
        	throw new M4CompilerError("SegmentationByPartitioning: Parameter 'HowManyPartitions' must be an integer!");
        }
    }
    
	/**
	 * @see edu.udo.cs.miningmart.m4.core.operator.Segmentation#setCSSegmentInfo(String, Columnset, int)
	 */
	public void setCSSegmentInfo(String inputMultiStepBranch, Columnset cs, int index)
		throws M4CompilerError
	{
		try {
			cs.addMultiStepBranchInfo(inputMultiStepBranch, MULTISTEPBRANCH_ATTRIBUTE, Integer.toString(index));
		}
   		catch (M4Exception m4e)
   		{   throw new M4CompilerError("M4 interface error in " + this.getName() + ": " + m4e.getMessage());  } 
	}
}
/*
 * Historie
 * --------
 *
 * $Log: SegmentationByPartitioning.java,v $
 * Revision 1.4  2006/04/11 14:10:11  euler
 * Updated license text.
 *
 * Revision 1.3  2006/04/06 16:31:10  euler
 * Prepended license remark.
 *
 * Revision 1.2  2006/03/23 11:13:45  euler
 * Improved exception handling.
 *
 * Revision 1.1  2006/01/03 09:54:21  hakenjos
 * Initial version!
 *
 */

/*
 * Old Historie
 * ------------
 *
 * Revision 1.13  2003/06/26 16:57:30  scholz
 * restructured operator to get along without the java stored procedure sampling method
 *
 * Revision 1.12  2002/12/02 14:07:05  euler
 * Bugfixes in Sampling.
 *
 * Revision 1.11  2002/12/02 08:37:56  euler
 * Bugfix.
 *
 * Revision 1.10  2002/10/25 13:43:26  euler
 * Changed Sql Definitions that FeatureConstruction
 * operators create. Adapted the Join operators.
 *
 * Revision 1.9  2002/10/08 18:07:55  scholz
 * Prepared code for parallel execution of multiple compile-Threads.
 * Calls to static fields like DB.m4Db were removed. Now each
 * Thread has its own DB object, reachable via
 * CompilerAccessLogic or Case.
 * The methods getCase() and getM4Db() were added to M4Object.
 * The static methods of Parameter now need an additional
 * parameter of type DB.
 * All direct calls from Operators to these Parameter methods were
 * removed.
 * All old load() and print() routines were removed.
 * The static calls to Print were removed. Now CompilerAccessLogic
 * references a valid Print object for the current Thread. This is
 * reachable via Case. The methods doPrint for messages and
 * Exceptions were added to M4Object.
 * The Print mechanism is not fully functional, yet.
 * A getStatus method was added to the Interface. It is not
 * functional yet for multiple Threads.
 *
 * Status: Compiles.
 *
 * Revision 1.8  2002/08/29 16:23:19  scholz
 * Unsegmentation support for MRFC.
 * Bugfixes (e.g. "Unsegment" now finally in repository ;-) )
 *
 * Revision 1.7  2002/08/28 19:25:05  scholz
 * First step for automatically loading parameters and checking
 * their validity:
 * Information stored in OP_PARAM_T is represented in an own
 * class and automatically loaded by class Operator.
 *
 * Unsegment:
 * The field CS_MSBRANCH (table COLUMNSET_T) is now
 * set during operator execution.
 * Each Columnset has the information attached, by which
 * segmentations (e.g.' item=19; ' or '(KMeans)=1' ) it was
 * created. This information is used by the Unsegment operator
 * to perform a UNION on subsets of all Columnsets for a
 * Concept. Additionally the segmentation value is restored
 * when "reversing" SegmentationStratified.
 * Status of Unsegment:
 * - Sources compile
 * - CS_MSBRANCH is set correctly, except for MRFC.
 *
 * Revision 1.6  2002/08/05 10:35:48  euler
 * Restructured the operator hierarchy: introduction of SingleCSOperator
 * and MultipleCSOperator. Changed this class accordingly.
 *
 * Revision 1.5  2002/05/21 12:51:01  euler
 * First beta test ok.
 *
 * Revision 1.4  2002/05/15 13:05:02  euler
 * First version that compiles.
 *
 * Revision 1.3  2002/05/15 10:36:43  euler
 * First version that compiles.
 *
 * Revision 1.2  2002/05/07 13:06:57  wiese
 * get+set methods
 *
 * Revision 1.1  2002/04/30 13:00:30  wiese
 * compileable version
 *
 * Revision 1.6  2002/04/19 15:23:12  wiese
 * Initialversion nach der uebergabe
 *
 */


