/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;

import edu.udo.cs.miningmart.m4.Column;
import edu.udo.cs.miningmart.m4.Step;
import edu.udo.cs.miningmart.m4.Value;
import edu.udo.cs.miningmart.exception.M4CompilerError;

/**
 * The class <i>ManualDiscretization</i> is abstract class
 * for the operators of type ManualDiscretization
 * 
 * @author Timm Euler, Martin Scholz
 * @version $Id: ManualDiscretization.java,v 1.5 2006/04/11 14:10:10 euler Exp $
 */
public abstract class ManualDiscretization extends Discretization  {

	protected static String INTERVAL_EXCLUSION = "E";
	protected static String INTERVAL_INCLUSION = "I";

	/** 
	 * Simply calls its superclass load method. 
	 */
	public void load(Step st) throws M4CompilerError {
		super.load(st);
		if (this.getIntervalStart().length != this.getIntervalEnd().length
			|| this.getIntervalEnd().length != this.getMapTo().length
			|| this.getMapTo().length != this.getStartIncExc().length
			|| this.getStartIncExc().length != this.getEndIncExc().length)
			throw new M4CompilerError(this.getName() + ": Parameter lists IntervalStart, "
					+ "IntervalEnd, MapTo, StartIncExc and EndIncExc sizes must be equal!");
	}
	
	/** 
	 * The operators Time/NumericalIntervalManualDiscretisation are not loopable
	 * any longer. Therefore the method <code>getCurrentLoopNumber()</code>
	 * is overwritten.
	 * 
	 * @return 0
	 */
	public int getCurrentLoopNumber() {
		return 0;
	}
	
    public Value getDefaultValue() throws M4CompilerError {
      return (Value)this.getSingleParameter("DefaultValue");
    }

    /**
     * The parameter IntervalStart is coordinated.
     */
    public Value[] getIntervalStart() throws M4CompilerError {
      return (Value[])this.getParameter("IntervalStart");
    }

    /**
     * The parameter IntervalEnd is coordinated.
     */
    public Value[] getIntervalEnd() throws M4CompilerError {
      return (Value[])this.getParameter("IntervalEnd");
    }

    /**
     * The parameter MapTo is coordinated.
     */
    public Value[] getMapTo() throws M4CompilerError {
      return (Value[])this.getParameter("MapTo");
    }

    /**
     * The parameter StartIncExc is coordinated.
     */
    public Value[] getStartIncExc() throws M4CompilerError {
      return (Value[])this.getParameter("StartIncExc");
    }

    /**
     * The parameter EndIncExc is coordinated.
     */
    public Value[] getEndIncExc() throws M4CompilerError {
      return (Value[])this.getParameter("EndIncExc");
    }
  
    /** Generates a sql-function to describe a new column.
     * This method generates a sql-function to describe a new
     * column. It is implemented in its subclasses 
     * @return the SQL-function call
     */
    public abstract String generateSQL(Column theTargetAttributeColumn) throws M4CompilerError;

	protected String getDiscTableName() {
	   return "M4_DT_" + this.getStep().getId() + "_" + this.getCurrentLoopNumber();
	}
	
	
	// convenience methods to access the information of one interval
	
	protected String getSmaller(int intervalNumber, boolean start) throws M4CompilerError {
		// use either the start or end boundary marker, according to the given boolean value:
		String incOrExc = (start ? this.getStartIncExc()[intervalNumber].getValue() : this.getEndIncExc()[intervalNumber].getValue());
		String parameterName = (start ? "StartIncExc" : "EndIncExc");
		if (incOrExc.equals(ManualDiscretization.INTERVAL_EXCLUSION)) {
			return " < ";
		}
		if (incOrExc.equals(ManualDiscretization.INTERVAL_INCLUSION)) {
			return " <= ";
		}
		throw new M4CompilerError("ManualDiscretization: Found unknown value for parameter '" +
		                          parameterName + "': " + incOrExc + 
		                          "; expected '" + ManualDiscretization.INTERVAL_EXCLUSION + 
		                          "' or '" + ManualDiscretization.INTERVAL_INCLUSION + "'!");
	}	
	
	protected String getGreater(int intervalNumber, boolean start) throws M4CompilerError {
		// use either the start or end boundary marker, according to the given boolean value:
		String incOrExc = (start ? this.getStartIncExc()[intervalNumber].getValue() : this.getEndIncExc()[intervalNumber].getValue());
		String parameterName = (start ? "StartIncExc" : "EndIncExc");
		if (incOrExc.equals(ManualDiscretization.INTERVAL_EXCLUSION)) {
			return " > ";
		}
		if (incOrExc.equals(ManualDiscretization.INTERVAL_INCLUSION)) {
			return " >= ";
		}
		throw new M4CompilerError("ManualDiscretization: Found unknown value for parameter '" +
		                          parameterName + "': '" + incOrExc + 
		                          "'; expected '" + ManualDiscretization.INTERVAL_EXCLUSION + 
		                          "' or '" + ManualDiscretization.INTERVAL_INCLUSION + "'!");
	}
}
/*
 * Historie
 * --------
 * 
 * $Log: ManualDiscretization.java,v $
 * Revision 1.5  2006/04/11 14:10:10  euler
 * Updated license text.
 *
 * Revision 1.4  2006/04/06 16:31:10  euler
 * Prepended license remark.
 *
 * Revision 1.3  2006/03/30 16:07:13  scholz
 * fixed author tags for release
 *
 * Revision 1.2  2006/03/23 11:13:45  euler
 * Improved exception handling.
 *
 * Revision 1.1  2006/01/03 09:54:20  hakenjos
 * Initial version!
 *
 */
