/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;

import java.sql.SQLException;

import edu.udo.cs.miningmart.storedProcedures.EMF;
import edu.udo.cs. miningmart.storedProcedures.TimeOperatorException;

import edu.udo.cs.miningmart.db.DB;
import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Column;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Value;

/**
 * @author Martin Scholz
 * @version $Id: ExponentialMovingFunction.java,v 1.5 2006/04/11 14:10:11 euler Exp $
 */
public class ExponentialMovingFunction extends TimeOperator {

    /* Some general settings: */
    // Parameter names:
    private static final String INPUTTIMEBA = "InputTimeBA";
    private static final String INPUTVALUEBA = "InputValueBA";
    private static final String OUTPUTTIMEBA = "OutputTimeBA";
    private static final String OUTPUTVALUEBA = "OutputValueBA";
    private static final String HEADWEIGHT = "HeadWeight";
    private static final String TAILWEIGHT = "TailWeight";
    private static final String DISTANCE = "Distance";

    private static final String procedureName      = "EMA";
    private static final String columnSetSuffix    = "_CS";

    protected String getColumnSetSuffix() { return ExponentialMovingFunction.columnSetSuffix; }

    public BaseAttribute getTimeBaIn() throws M4CompilerError {
    	return (BaseAttribute) this.getSingleParameter(INPUTTIMEBA);
    }
    
    public BaseAttribute getValueBaIn() throws M4CompilerError {
       	return (BaseAttribute) this.getSingleParameter(INPUTVALUEBA);
    }
    	
    public BaseAttribute getTimeBaOut() throws M4CompilerError {
       	return (BaseAttribute) this.getSingleParameter(OUTPUTTIMEBA);
    }

    public BaseAttribute getValueBaOut() throws M4CompilerError {
       	return (BaseAttribute) this.getSingleParameter(OUTPUTVALUEBA);
    }

    public Value getHeadWeight() throws M4CompilerError {
    	return (Value) this.getSingleParameter(HEADWEIGHT);
    }

    public Value getTailWeight() throws M4CompilerError {
    	return (Value) this.getSingleParameter(TAILWEIGHT);
    }

    public Value getDistance() throws M4CompilerError {
    	return (Value) this.getSingleParameter(DISTANCE);
    }


    /* -------- Create and link columns: -------- */
    protected void createJavaColumns(Columnset newCS) throws SQLException, M4CompilerError 
    {
		try {
			final Column timeColumnForType = this.getTimeBaIn().getColumn(0);
			createM4Column(getTimeBaOut(), newCS, timeColumnForType);

			final Column valueColumnForType = this.getValueBaIn().getColumn(0);
			createM4Column(getValueBaOut(), newCS, valueColumnForType);
		}
   		catch (M4Exception m4e)
   		{   throw new M4CompilerError("M4 interface error in " + this.getName() + ": " + m4e.getMessage());  }
    }

    // ------------------------------------------------------------------------------

    /*
     * ----- Method invokes the following stored procedure: -----
     *
     *  EMA( InputColumnSet  VARCHAR2,
     *       TimeAttribIn    VARCHAR2,
     *       ValueAttribIn   VARCHAR2,
     *       TargetColumnSet VARCHAR2,
     *       TimeAttribOut   VARCHAR2,
     *       ValueAttribOut  VARCHAR2,
     *       headWeight      NUMBER,
     *       tailWeight      NUMBER,
     *       distance        NUMBER)
     */
    protected void runStoredProcedure(String newColumnsetName)
    	throws SQLException, M4CompilerError, TimeOperatorException
    {

		/* Check if values are numeric: */
		this.valueIsDouble(getHeadWeight(), HEADWEIGHT);
		this.valueIsDouble(getTailWeight(), TAILWEIGHT);
		this.valueIsLong(getDistance(),     DISTANCE);

		/* Check if input BaseAttributes have a column: */
		this.baseattribHasColumns(getTimeBaIn(),  INPUTTIMEBA);
		this.baseattribHasColumns(getValueBaIn(), INPUTVALUEBA);


		String[] parameters = {
		    //Input:
	    	getQuotedColSetNameforCon(getInputConcept()), // Input Columnset
		    getQuotedColNameforInpBA(getTimeBaIn()),      // Time BA
		    getQuotedColNameforInpBA(getValueBaIn()),     // Value BA
	    	// Output:
		    DB.quote(newColumnsetName),                 // Output Columnset
		    getQuotedColNameforOutBA(getTimeBaOut()),   // BA: Time for output
	    	getQuotedColNameforOutBA(getValueBaOut()),  // BA: output values
		    // Specific Parameters:
		    getHeadWeight().getValue().trim(), // head weight
	    	getTailWeight().getValue().trim(), // tail weight
		    getDistance().getValue().trim()    // Distance (step size)
		};

		if (storedProceduresAvailable()) {
			this.executeDbProcedure(ExponentialMovingFunction.procedureName, parameters);
		}
		else {	
				EMF.dbEMA( this.getM4Db(),
					       unquote(parameters[0]),
					       unquote(parameters[1]),
					       unquote(parameters[2]),
					       unquote(parameters[3]),
					       unquote(parameters[4]),
					       unquote(parameters[5]),
						   Double.parseDouble(parameters[6]),
						   Double.parseDouble(parameters[7]),
						   Integer.parseInt(parameters[8])
						 );	
		}
    } // end private void runStoredProcedure

}
/*
 * Historie
 * --------
 *
 * $Log: ExponentialMovingFunction.java,v $
 * Revision 1.5  2006/04/11 14:10:11  euler
 * Updated license text.
 *
 * Revision 1.4  2006/04/06 16:31:10  euler
 * Prepended license remark.
 *
 * Revision 1.3  2006/03/30 16:07:13  scholz
 * fixed author tags for release
 *
 * Revision 1.2  2006/03/23 11:13:45  euler
 * Improved exception handling.
 *
 * Revision 1.1  2006/01/03 09:54:21  hakenjos
 * Initial version!
 *
 */
