/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Vector;

import edu.udo.cs.miningmart.db.DB;
import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Concept;
import edu.udo.cs.miningmart.m4.EstimatedStatistics;
import edu.udo.cs.miningmart.m4.Step;
import edu.udo.cs.miningmart.m4.utils.Print;

/**
 * This class is the superclass for evaluation operators. They do not
 * have an output on the conceptual level, ie no output concept or output
 * attribute. They may be used to evaluate a data mining operator. They
 * compare the values of two columns.
 * 
 * @author Timm Euler
 * @version $Id: EvaluateResults.java,v 1.7 2006/04/11 14:10:11 euler Exp $
 */
public abstract class EvaluateResults extends ExecutableOperator
{
	private Columnset theColumnSet;
    private String inputColumnSetName, actualColumnName, predictedColumnName;
    private double[] actualvalue,predictedvalue;

	/**
	 * Abstract method to be implemented by all subclasses. The values of two
	 * columns are compared to compute an error. The "true" values
	 * of a column as well as the learned/predicted ones are given. The result
	 * should be an error value of some description.
	 * 
	 * @param actualvalue An array with the true values.
	 * @param predicted An array with the predicted values.
	 * @return an error value
	 */
    public abstract double computeResult(double actualvalue[] ,double predicted[]) throws M4CompilerError;

    private String generateSQL() throws M4CompilerError //(Column columnToFill)
    {
    	try {
	        theColumnSet = this.getInputConcept().getCurrentColumnSet();
		 	inputColumnSetName = theColumnSet.getSchema()+"."+theColumnSet.getName();
	        actualColumnName = this.getActualValueAttribute().getCurrentColumn().getSQLDefinition();
		 	predictedColumnName = this.getPredictedValueAttribute().getCurrentColumn().getSQLDefinition();

			String callSQL = "SELECT "+ actualColumnName + "," + predictedColumnName + " FROM " + inputColumnSetName;

			return callSQL;
    	}
   		catch (M4Exception m4e)
   		{   throw new M4CompilerError("M4 interface error in " + this.getName() + ": " + m4e.getMessage());  }
    }

	/**
	 * @see edu.udo.cs.miningmart.operator.Operator#createStatement
	 */
    public void createStatement(boolean lazy) throws SQLException, M4CompilerError
    {
		// lazy is ignored
        String query = generateSQL();
        this.doPrint(Print.DB_READ, "DB Query: " + query);
        ResultSet rs = null;
        try {
        	rs = this.getM4Db().executeBusinessSqlRead(query);
			Vector actval=new Vector();
			Vector predval=new Vector();

		    while (rs.next()) {
				actval.add(new Double(rs.getDouble(1))); // actualValue
				predval.add(new Double(rs.getDouble(2))); // predictedValue
		    }
	        actualvalue = new double[actval.size()];
    	    predictedvalue = new double[actval.size()];
	    	for (int i=0; i<actualvalue.length; i++) {
	        	actualvalue[i] = ((Double)actval.get(i)).doubleValue();
				predictedvalue[i] = ((Double)predval.get(i)).doubleValue();
		    }
        	double error=0;
	        error=computeResult(actualvalue,predictedvalue);
    	    this.doPrint(Print.MAX, "Compared " + this.getActualValueAttribute().getName() +
        	                        " and " + this.getPredictedValueAttribute().getName() +
            	                    "; computed this error: "+error);
        }
        finally {
        	DB.closeResultSet(rs);	
        }
   } 

	/**
	 * @see edu.udo.cs.miningmart.operator.ExecutableOperator#writeResults()
	 */
    public void writeResults() throws SQLException, M4CompilerError
    {   }

	/**
	 * @see edu.udo.cs.miningmart.m4.ExecutableOperator#compileStatement()
	 */
    public void compileStatement() throws SQLException, M4CompilerError
    {   }

    /**
     * @see edu.udo.cs.miningmart.operator.ExecutableOperator#estimateStatistics(Step)
     */
    public EstimatedStatistics estimateStatistics(Step theStep) throws M4Exception {
    	return null;
    }
    
	/**
	 * Gets the actualValueAttribute.
	 * 
	 * @return Returns a BaseAttribute
	 */
	public BaseAttribute getActualValueAttribute() throws M4CompilerError {
		return (BaseAttribute) this.getSingleParameter("TheTargetValueAttribute");
	}

	/**
	 * Gets the input Concept.
	 * 
	 * @return Returns a Concept
	 */
	public Concept getInputConcept() throws M4CompilerError {
		return (Concept) this.getSingleParameter("TheInputConcept");
	}

	/**
	 * Gets the predictedValueAttribute.
	 * 
	 * @return Returns a BaseAttribute
	 */
	public BaseAttribute getPredictedValueAttribute() throws M4CompilerError {
		return (BaseAttribute) this.getSingleParameter("ThePredictedValueAttribute");
	}
}
/*
 * Historie
 * --------
 *
 * $Log: EvaluateResults.java,v $
 * Revision 1.7  2006/04/11 14:10:11  euler
 * Updated license text.
 *
 * Revision 1.6  2006/04/06 16:31:11  euler
 * Prepended license remark.
 *
 * Revision 1.5  2006/03/30 16:07:12  scholz
 * fixed author tags for release
 *
 * Revision 1.4  2006/03/23 11:13:44  euler
 * Improved exception handling.
 *
 * Revision 1.3  2006/01/18 16:58:58  euler
 * Added some basic estimations of statistics.
 * Will need improvements.
 *
 * Revision 1.2  2006/01/05 10:27:38  hakenjos
 * Removed Javadoc Warnings!
 *
 * Revision 1.1  2006/01/03 09:54:21  hakenjos
 * Initial version!
 *
 */
