/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;

import edu.udo.cs.miningmart.db.DB;
import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Column;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Concept;
import edu.udo.cs.miningmart.m4.PrimaryKey;
import edu.udo.cs.miningmart.m4.Value;

/**
 * This operator is able to convert a view with no key and maybe multiple
 * occurences of tuples into a view with a single primary key attribute.
 * Elimination of multiple tuple occurences can be selected as a parameter.
 * 
 * @author Martin Scholz
 * @version $Id: CreatePrimaryKey.java,v 1.4 2006/04/11 14:10:11 euler Exp $
 */
public class CreatePrimaryKey extends SingleCSOperator {
	
	/**
	 * @see SingleCSOperator#getTypeOfNewColumnSet()
	 */
	public String getTypeOfNewColumnSet() {
		return Columnset.CS_TYPE_VIEW;
	}
	
	/** 
	 * The super method is enhanced by creating a new Column for the key.
	 * The key is realized as a virtual Column, so the return value of the
	 * super method is not changed.
	 * 
	 * @see miningmart.operator.ConceptOperator#generateColumns(Columnset)
	 */
    protected String generateColumns(Columnset csForOutputConcept)
    	throws M4CompilerError
    {
    	String selectPart = super.generateColumns(csForOutputConcept);
    	String columnName = this.getNewKeysColumnName();
		String sqlDef = this.getNewKeysSqlDefinition();

		try {	
	    	Column newKey = (Column) this.getM4Db().createNewInstance(edu.udo.cs.miningmart.m4.core.Column.class);
    		newKey.setName(columnName);
    		newKey.setColumnDataTypeName("KEY");
	    	newKey.setColumnset(csForOutputConcept);
	    	
			// Realized as a virtual column:
			newKey.setSQLDefinition(sqlDef);
			
			// Finally add new Column to the BaseAttribute's Columns:
			this.getPrimaryKey().addColumn(newKey);
			
			// TE 15.10.2004: Also store primary key info in M4:
			PrimaryKey pk = (PrimaryKey) this.getM4Db().createNewInstance(edu.udo.cs.miningmart.m4.core.PrimaryKey.class);
			pk.addColumn(newKey);
			pk.setColumnset(csForOutputConcept);
		}
		catch (M4Exception e) {
			throw new M4CompilerError(
				"CreatePrimaryKey.generateColumns: M4Exception when trying to create the PrimaryKey column!\n"
				+ e.getMessage());	
		}

		// We do not change the select-part, because the ID attribute is a virtual column!
		return selectPart;
    }
	
	/**
	 * @see SingleCSOperator#generateSQLDefinition(String)
	 */
	public String generateSQLDefinition(String selectPart) throws M4CompilerError {
		
		Concept concept = this.getInputConcept();		
		Columnset cs;
		
		try {		
			if (concept == null || (cs = concept.getCurrentColumnSet()) == null) {
				throw new M4CompilerError("Operator 'CreatePrimaryKey': Found no input Concept/Columnset!");
			}
			
			// TE 3.1.05: what's this?? (commented two lines out)
			// String tableName = this.getNewCSName();
			// this.getM4Db().dropBusinessTable(tableName); // try to drop the table first if it exists

			String sql = "(SELECT " + (this.getAllowDuplicates() ? "" : "DISTINCT ")
					   + selectPart + " FROM " + cs.getSchemaPlusName() + ")";

			return sql;
		}
		catch (M4Exception e) {
			throw new M4CompilerError(
				"Operator 'CreatePrimaryKey': M4Exception caught during createStatement:\n"
				+ e.getMessage());
		}
	}
	
	/**
	 * @see ConceptOperator#mustCopyFeature(String)
	 */
	protected boolean mustCopyFeature(String nameOfFeature) throws M4CompilerError {
		return true;
	}	

	/** Encapsulate the mapping from BaseAttribute to new Column name. */
	private String getNewKeysColumnName() throws M4CompilerError {
		return this.getPrimaryKey().getName();
	}

	/**
	 * This method returns the definition of the new key column.
	 * If there is no definition available for the DBMS at hand, then
	 * this method throws an M4CompilerError.
	 */
	private String getNewKeysSqlDefinition() throws M4CompilerError {
		try {
			// if the input concept is a table, we can use Oracle or Postgres;
			// but if it is a view, only Oracle works so far:
			if ( ( ! this.getInputConcept().getCurrentColumnSet().getType().equals(Columnset.CS_TYPE_TABLE)) 
			     && 
			     (this.getBusinessDbms() != DB.ORACLE) )
			{
				throw new M4CompilerError(
					"Sorry, operator '" + this.getName() + "' has not yet been implemented for input VIEWS for"
					+ " any other DBMS than Oracle!");
			}
		}
		catch (M4Exception m4e)
		{   throw new M4CompilerError("M4 Exception caught in '" + this.getName() + "': " + m4e.getMessage());  }
		
		boolean useDbmsOfBusinessDataSchema = true;
		return this.getM4Db().getUniqueRowIdentifier(useDbmsOfBusinessDataSchema);
	}

	// *** specific parameters of this operator: ***

	/**
	 * Getter for the parameter &quot;AllowDuplicates&quot; 
	 * @return a <code>boolean</code>, <code>true</code> means that duplicates are allowed,
	 * the default is <code>false</code>.
	 */
	public boolean getAllowDuplicates() throws M4CompilerError {
		Value value = (Value) this.getSingleParameter("AllowDuplicates");
		if (value != null) {
			String s = value.getValue();
			if (s != null && s.equalsIgnoreCase("true")) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Getter for the parameter &quot;PrimaryKey&quot; 
	 * @return a <code>BaseAttribute</code>
	 */
	public BaseAttribute getPrimaryKey() throws M4CompilerError {
		return (BaseAttribute) this.getSingleParameter("PrimaryKey");
	}
}
/*
 * Historie
 * --------
 *
 * $Log: CreatePrimaryKey.java,v $
 * Revision 1.4  2006/04/11 14:10:11  euler
 * Updated license text.
 *
 * Revision 1.3  2006/04/06 16:31:11  euler
 * Prepended license remark.
 *
 * Revision 1.2  2006/03/23 11:13:44  euler
 * Improved exception handling.
 *
 * Revision 1.1  2006/01/03 09:54:22  hakenjos
 * Initial version!
 *
 */
