/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;

import edu.udo.cs.miningmart.db.DB;
import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Column;
import edu.udo.cs.miningmart.m4.RelationalDatatypes;
import edu.udo.cs.miningmart.m4.Value;

/**
 * This operator creates new binary attributes that realise a boolean flag
 * indicating for each value of the target attribute whether it
 * occurs in that row/entity.
 *
 * @author Timm Euler
 * @version $Id: Binarify.java,v 1.4 2006/04/11 14:10:12 euler Exp $
 */
public class Binarify extends FeatureConstruction {

	// Parameter names for this Operator
	public static final String PARAMETER_VALUES = "TargetValue";
	public static final String PARAMETER_TARGETATTR = "TheTargetAttribute";

    public String generateSQL(Column targetAttributeColumn) throws M4CompilerError {
    	
    	Value v = (Value) this.getSingleParameter(PARAMETER_VALUES, this.getCurrentLoopNumber());
    	if (v == null || v.getValue() == null || v.getValue().equals("")) {
    		throw new M4CompilerError("Binarify, loop nr. " + this.getCurrentLoopNumber() + ": could not read target value!");
    	}
    	String targetValue = v.getValue();
    	
    	try {
    		String when = targetAttributeColumn.getSQLPlusLocation() + " = ";
    		// check if the target Column is string:
    		boolean targetColNominal = (targetAttributeColumn.getColumnDataTypeName().equalsIgnoreCase(RelationalDatatypes.RELATIONAL_DATATYPE_STRING));
    		when += (targetColNominal ? DB.quote(targetValue) : targetValue);
    		
    		String colDef = "(CASE WHEN " +
	                    	when + 
							" THEN 1 ELSE 0 END)";
    		return colDef;
    	} 
    	catch (M4Exception m4e) {
    		throw new M4CompilerError("Binarify: M4Exception caught when accessing target attribute column '" + targetAttributeColumn.getName() + "': " + m4e.getMessage());
    	}
    }
    

	/**
	 * All FeatureConstruction operators have a target attribute which 
	 * is different from loop to loop. Only this operator, Binarify,
	 * has got the same target attribute in all loops. So the super class
	 * method is overwritten.
	 * 
	 * @param loopNr the number of the loop for which the target attribute is wanted
	 * @return the target attribute for this operator 
	 */
    public BaseAttribute getTheTargetAttribute(int loopNr) throws M4CompilerError {
		return (BaseAttribute) this.getSingleParameter(PARAMETER_TARGETATTR);
    }
}
/*
 * Historie
 * --------
 * 
 * $Log: Binarify.java,v $
 * Revision 1.4  2006/04/11 14:10:12  euler
 * Updated license text.
 *
 * Revision 1.3  2006/04/06 16:31:11  euler
 * Prepended license remark.
 *
 * Revision 1.2  2006/03/23 11:13:45  euler
 * Improved exception handling.
 *
 * Revision 1.1  2006/03/17 17:06:39  euler
 * *** empty log message ***
 *
 */
