/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4.utils;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.Iterator;
import java.util.Vector;

/**
 * A <code>MultipleOutputStream</code> holds several <code>OutputStream</code>s.
 * Each method called by this <code>OutputStream</code> is forwarded to the set of <code>OutputStream</code>s.
 * So you can write something to <code>System.out</code> and to a file.
 * @author Daniel Hakenjos
 * @version $Id: MultipleOutputStream.java,v 1.3 2006/04/11 14:10:10 euler Exp $
 */
public class MultipleOutputStream extends OutputStream{

	private Vector streams;

	/**
	 * Inits this <code>MultipleOutputStream</code> containing no <code>OutputStream</code>.
	 */
	public MultipleOutputStream(){
		streams=new Vector();
		streams.trimToSize();
	}
	
	/**
	 * Inits this <code>MultipleOutputStream</code> only containing th specified <code>OutputStream</code>.
	 */
	public MultipleOutputStream(OutputStream stream){
		streams=new Vector();
		streams.add(stream);
		streams.trimToSize();
	}
	
	/**
	 * Inits this <code>MultipleOutputStream</code> with the specified <code>Collection</code> of <code>OutputStream</code>s.
	 */
	public MultipleOutputStream(Collection outputStreams){
		streams=new Vector();
		Iterator iter=outputStreams.iterator();
		Object stream;
		while (iter.hasNext()){
			stream = iter.next();
			if (iter instanceof OutputStream){
				streams.add(stream);
			}
		}			
		streams.trimToSize();
	}


	
	/**
	 * @see java.io.OutputStream#close()
	 */
	public void close() throws IOException {
		OutputStream stream;
		for (int i=0; i<streams.size();i++){
			stream=(OutputStream) streams.get(i);
			stream.close(); 
		}		
	}

	/**
	 * @see java.io.OutputStream#flush()
	 */
	public void flush() throws IOException {
		OutputStream stream;
		for (int i=0; i<streams.size();i++){
			stream=(OutputStream) streams.get(i);
			stream.flush(); 
		}		
	}

	/**
	 * @see java.io.OutputStream#write(byte[], int, int)
	 */
	public void write(byte[] b, int off, int len) throws IOException {
		OutputStream stream;
		for (int i=0; i<streams.size();i++){
			stream=(OutputStream) streams.get(i);
			stream.write(b, off, len);
		}		
	}

	/**
	 * @see java.io.OutputStream#write(byte[])
	 */
	public void write(byte[] b) throws IOException {
		OutputStream stream;
		for (int i=0; i<streams.size();i++){
			stream=(OutputStream) streams.get(i);
			stream.write(b);
		}		
	}

	/**
	 * @see java.io.OutputStream#write(int)
	 */
	public void write(int b) throws IOException {
		OutputStream stream;
		for (int i=0; i<streams.size();i++){
			stream=(OutputStream) streams.get(i);
			stream.write(b);
		}		
	}

	/**
	 * Gets the number of <code>OutputStream</code>s.
	 */
	public int getNumberOfOutputStreams(){
		return streams.size();
	}
	

	/**
	 * Gets the <code>OutputStream</code> by its index.
	 * @return null if the index is wrong
	 */
	public OutputStream getOutputStream(int index){
		if (index<0)
			return null;
		if (index>=streams.size())
			return null;
		return (OutputStream) streams.get(index);
	}	
	
	/**
	 * Adds the <code>OutputStream</code>.
	 */
	public void addOutputStream(OutputStream stream){
		streams.add(stream);
	}

	/**
	 * Removes the <code>OutputStream</code>.
	 */	
	public boolean removeOutputStream(OutputStream stream){
		return streams.remove(stream);
	}

	/**
	 * Checks wether the <code>OutputStream</code> is available or not.
	 */	
	public boolean containsOutputStream(OutputStream stream){
		return streams.contains(stream);
	}
}
/*
$Log: MultipleOutputStream.java,v $
Revision 1.3  2006/04/11 14:10:10  euler
Updated license text.

Revision 1.2  2006/04/06 16:31:10  euler
Prepended license remark.

Revision 1.1  2006/01/03 09:54:04  hakenjos
Initial version!

*/
