/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4.core;

import java.util.Collection;
import java.util.Iterator;

import edu.udo.cs.miningmart.db.DB;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.ForeignKeyLink;
import edu.udo.cs.miningmart.m4.utils.M4Info;
import edu.udo.cs.miningmart.m4.utils.M4InfoEntry;
import edu.udo.cs.miningmart.m4.utils.Print;
import edu.udo.cs.miningmart.m4.utils.XmlInfo;

/**
 * Each instance of this class represents an entry
 * in the table KEYMEMBER_T. Several instances of this
 * class may belong to the same <code>Key</code>.
 * 
 * @see edu.udo.cs.miningmart.m4.core.Key
 * 
 * @author Timm Euler
 * @version $Id: KeyMember.java,v 1.4 2006/04/11 14:10:14 euler Exp $
 */
public class KeyMember extends M4Data implements XmlInfo, ForeignKeyLink {
	
	// Database constants for the keymember table
	public static final String M4_TABLE_NAME        = "keymember_t";
	public static final String ATTRIB_MEM_ID        = "km_id";
	public static final String ATTRIB_MEM_HEAD_ID   = "km_khid";
	public static final String ATTRIB_MEM_POSITION  = "km_pos";
	public static final String ATTRIB_MEM_PK_COLUMN = "km_pkcolid";
	public static final String ATTRIB_MEM_FK_COLUMN = "km_fkcolid";
	
	// instance fields
	private Key myKey; // either a ForeignKey or a PrimaryKey
	private Column foreignKeyColumn, primaryKeyColumn;
	private int myPosition;
	
	/** Cache for getM4Info() */
	public static M4Info m4Info = null;

	/** @see M4Table.getM4TableName() */
	public String getM4TableName() {
		return M4_TABLE_NAME;	
	}

	/** @see M4Table.getIdAttributeName() */
	public String getIdAttributeName() {
		return ATTRIB_MEM_ID;
	}

	/** @see M4Table.getM4Info() */
	public M4Info getM4Info() 
	{
		if (m4Info == null) 
		{
		    M4InfoEntry[] m4i = {
				new M4InfoEntry(ATTRIB_MEM_ID,        "getId",               "setId",               long.class,  NOT_NULL),
				new M4InfoEntry(ATTRIB_MEM_POSITION,  "getThePosition",      "setThePosition",      int.class,   NOT_NULL),
				new M4InfoEntry(ATTRIB_MEM_HEAD_ID,   "getTheKey",           "setTheKey",           edu.udo.cs.miningmart.m4.Key.class,   NOT_NULL),
				new M4InfoEntry(ATTRIB_MEM_PK_COLUMN, "getPrimaryKeyColumn", "primitiveSetPrimaryKeyColumn", edu.udo.cs.miningmart.m4.Column.class),
				new M4InfoEntry(ATTRIB_MEM_FK_COLUMN, "getForeignKeyColumn", "primitiveSetForeignKeyColumn", edu.udo.cs.miningmart.m4.Column.class)
		  	};
		    m4Info = new M4Info(m4i);
		}
		return m4Info;
	}

	/** Cache for getXmlInfo() */
	private static M4Info xmlInfo = null;

	/** @see XmlInfo.getXmlInfo() */
	public M4Info getXmlInfo() {
		if (xmlInfo == null) {
			M4InfoEntry[] m4i = {
				new M4InfoEntry("Position",         "getThePosition",      "setThePosition",      int.class),
				new M4InfoEntry("Key",              "getTheKey",           "setTheKey",           edu.udo.cs.miningmart.m4.Key.class),
				new M4InfoEntry("PrimaryKeyColumn", "getPrimaryKeyColumn", "setPrimaryKeyColumn", edu.udo.cs.miningmart.m4.Column.class),
				new M4InfoEntry("ForeignKeyColumn", "getForeignKeyColumn", "setForeignKeyColumn", edu.udo.cs.miningmart.m4.Column.class),
				new M4InfoEntry("Docu",             "getDocumentation",    "setDocumentation",    String.class)
			};
			xmlInfo = new M4Info(m4i);
		}
		return xmlInfo;
	}
	
	/**
	 * Constructor for KeyMember.
	 * 
	 * @param m4Db The DB object
	 */
	public KeyMember(DB m4Db)
	{	super(m4Db);	}

	/**
	 * Used to print information about this object.
	 * 
	 * @see edu.udo.cs.miningmart.m4.core.M4Object#print
	 */
    public void print()
    {   this.doPrint(Print.M4_OBJECT, "KeyMember: Id = " + myId + ";");    }
	
	/**
	 * @see edu.udo.cs.miningmart.m4.core.M4Data#getObjectsInNamespace(Class)
	 */
	protected Collection getObjectsInNamespace(Class typeOfObjects) throws M4Exception {
		return null;
	}
        
	/**
	 * Returns the ForeignKey Column.
	 * 
	 * @return The foreign key column of this key member
	 */
	public edu.udo.cs.miningmart.m4.Column getForeignKeyColumn()
	{	return foreignKeyColumn;   }

	/**
	 * Returns the Key this Keymember belongs to
	 * 
	 * @return the Keyhead
	 */
	public edu.udo.cs.miningmart.m4.Key getTheKey()
	{	return myKey;	}

	/**
	 * Returns this keymember's position in the keyhead.
	 * 
	 * @return the position
	 */
	public int getThePosition()
	{	return myPosition;	}

	/**
	 * Returns the Primary Key Column.
	 * 
	 * @return The primary key column of this keymember
	 */
	public edu.udo.cs.miningmart.m4.Column getPrimaryKeyColumn()
	{	return primaryKeyColumn;	}

	/**
	 * Sets the foreign Key Column.
	 * 
	 * @param foreignKeyColumn The foreign Key Column to set
	 */
	public void setForeignKeyColumn(edu.udo.cs.miningmart.m4.Column foreignKeyColumn) throws M4Exception {	
		Column.colFkMem.updateReferenceTo(this, foreignKeyColumn);
	}

	/** Primitive setter, do not use it! */
	public void primitiveSetForeignKeyColumn(edu.udo.cs.miningmart.m4.Column column) {	
		this.setDirty();
		this.foreignKeyColumn = (Column) column;
	}

	/**
	 * Sets the Keyhead for this keymember
	 * 
	 * @param newKey The keyhead
	 */
	public void setTheKey(edu.udo.cs.miningmart.m4.Key newKey) throws M4Exception {	
		Key.key2member.checkNameExists(this, newKey);
		Key.key2member.updateReferenceTo(this, newKey);	
	}

	/**
	 * Sets the position.
	 * 
	 * @param newPosition The position to set
	 */
	public void setThePosition(int newPosition)
	{	
		this.setDirty();	
		this.myPosition = newPosition;
	}

	/**
	 * Sets the primary Key Column.
	 * 
	 * @param primaryKeyColumn The primary Key Column to set
	 */
	public void setPrimaryKeyColumn(edu.udo.cs.miningmart.m4.Column primaryKeyColumn) throws M4Exception {
		Column.colPkMem.updateReferenceTo(this, primaryKeyColumn);
	}

	/** Primitive setter, do not use it! */
	public void primitiveSetPrimaryKeyColumn(edu.udo.cs.miningmart.m4.Column column) {
		this.setDirty();
		this.primaryKeyColumn = (Column) column;		
	}

    /**
	 * Primitive setter, do not use.
	 * @param k the <code>Key</code> to be set
	 * */
    public void primitiveSetKey(Key k) {
    	this.setDirty();
    	this.myKey = k;
    }
    
   	/** @see M4Data#removeAllM4References() */
	protected void removeAllM4References() throws M4Exception {
		this.setForeignKeyColumn(null);
		this.setPrimaryKeyColumn(null);
		this.setTheKey(null);
		this.removeDocObject();
	}
}
/*
 * Historie
 * --------
 * 
 * $Log: KeyMember.java,v $
 * Revision 1.4  2006/04/11 14:10:14  euler
 * Updated license text.
 *
 * Revision 1.3  2006/04/06 16:31:13  euler
 * Prepended license remark.
 *
 * Revision 1.2  2006/01/06 16:25:04  euler
 * Updates and bugfixes in the delete-Mechanism for M4Data objects.
 *
 * Revision 1.1  2006/01/03 09:54:17  hakenjos
 * Initial version!
 *
 */
