/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4;

import java.util.Collection;

import edu.udo.cs.miningmart.exception.M4Exception;

/**
 * A ForeignKey models a foreign key relationship between two
 * Columnsets. One of the Columnsets has a foreign key that 
 * refers to the primary key of the other Columnset. So each
 * ForeignKey has a PrimaryKeyColumnset and a ForeignKeyColumnset.
 * In both Columnsets, several Columns may be involved in the
 * key; in this case, the ForeignKey maps each Column of the key of 
 * the PrimaryKeyColumnset to the corresponding Column of the key
 * of the ForeignKeyColumnset.
 * 
 * A 1:n Relation has one ForeignKey to model its links on the 
 * relational level. An n:m Relation uses TWO ForeignKeys,
 * to model the relationship between the Columnset 
 * of the FromConcept and the cross table on the one hand, and
 * the relationship between the Columnset of the ToConcept and
 * the cross table on the other hand.
 * 
 * @author Daniel Hakenjos, Timm Euler
 * @version $Id: ForeignKey.java,v 1.4 2006/04/11 14:10:12 euler Exp $
 */
public interface ForeignKey extends Key {

	/**
	 * Getter method.
	 * 
	 * @return The Relation this Key is associated with, if any.
	 */
	public Relation getRelation() throws M4Exception;

	/**
	 * Setter method.
	 * 
	 * @param rel The Relation this ForeignKey is to be associated with
	 * @param isFrom TRUE iff this ForeignKey is the fromKey of that Relation,
	 *        FALSE if it is the toKey. (An n:m Relation needs both a fromKey
	 *        and a toKey. A 1:n Relation needs only a fromKey.)
	 */
	public void setRelation(Relation rel, boolean isFrom) throws M4Exception;
	
    /**
     * Adds a Column Link to this ForeignKey. A ForeignKey may consist of one or more
     * Columns in each of the two Columnsets. The Columns in the two Columnsets
     * will be mapped to each other.
     * 
     * @param fkColumn The foreign key Column to add to this ForeignKey.
     * @param pkColumn The primary key Column that is referenced by fkColumn.
     */
    public void addColumnLink(Column fkColumn, Column pkColumn) throws M4Exception;
    
    /**
     * Removes a Column Link from this ForeignKey. The name is used to refer
     * the Column of the foreign key Columnset, but the corresponding Column 
     * of the primary key Columnset is also removed from this ForeignKey
     * (all Columns remain in their Columnsets, of course).
     */
    public void removeColumnLink(String foreignKeyColumnName) throws M4Exception;
    
    /**
     * Empties this ForeignKey of all Column Links.
     */
    public void removeAllColumnLinks() throws M4Exception;    
    
    /**
     * Returns TRUE iff the given Column acts as a foreign key Column in
     * this ForeignKey.
     * 
     * @param theColumn the given Column
     * @return true iff it is a foreign key column
     * @throws M4Exception
     */
	public boolean isColumnContainedInForeignColumns(Column theColumn) throws M4Exception;

    /**
     * Returns TRUE iff the given Column acts as a primary key Column in
     * this ForeignKey.
     * 
     * @param theColumn the given Column
     * @return true iff it is a primary key column
     * @throws M4Exception
     */
	public boolean isColumnContainedInPrimaryColumns(Column theColumn) throws M4Exception;

	/**
	 * Returns the primary key column that is associated with the given foreign
	 * key column, if the latter exists in this ForeignKey. Otherwise NULL is
	 * returned.
	 * 
	 * @param theForeignKeyColumn the given foreign key column
	 * @return the associated primary key column or NULL
	 * @throws M4Exception
	 */
	public Column getPrimaryForForeignColumn(Column theForeignKeyColumn) throws M4Exception;

	/**
	 * Returns the foreign key column that is associated with the given primary
	 * key column, if the latter exists in this ForeignKey. Otherwise NULL is
	 * returned.
	 * 
	 * @param thePrimaryKeyColumn the given primary key column
	 * @return the associated foreign key column or NULL
	 * @throws M4Exception
	 */
	public Column getForeignForPrimaryColumn(Column thePrimaryKeyColumn) throws M4Exception;
	
	/**
	 * Getter method.
	 * 
	 * @return The columnset of the primary key of this ForeignKey
	 */
	public Columnset getPrimaryKeyColumnset() throws M4Exception;

	/**
	 * Setter method.
	 * 
	 * @param cs The new columnset of the primary key of this ForeignKey
	 */
	public void setPrimaryKeyColumnset(Columnset cs) throws M4Exception;

	/**
	 * Getter method.
	 * 
	 * @return The columnset of the foreign key of this ForeignKey
	 */
	public Columnset getForeignKeyColumnset() throws M4Exception;

	/**
	 * Setter method.
	 * 
	 * @param cs The new columnset of the foreign key of this ForeignKey
	 */
	public void setForeignKeyColumnset(Columnset cs) throws M4Exception;
	
	/**
	 * This method returns a <code>Collection</code> of <code>ForeignKeyLink</code>s.
	 * Each <code>ForeignKeyLink</code> links one Column of the foreign Columnset
	 * to the corresponding Column of the primary Columnset.
	 * 
	 * @return a <code>Collection</code> of <code>ForeignKeyLink</code>s
	 * @throws M4Exception
	 */
	public Collection getAllColumnLinks() throws M4Exception;
	
	/**
	 * Convenience method used by classes that assume that each ForeignKey
	 * link involves only one Column in each of the two Columnsets. 
	 * This method simply returns the first link in the hope that there 
	 * is only one.
	 * 
	 * @return a ForeignKeyLink
	 * @throws M4Exception
	 */
	public ForeignKeyLink getFirstColumnLink() throws M4Exception;
	
	public void removeForeignKeyColumnset() throws M4Exception;
	
	public void removePrimaryKeyColumnset() throws M4Exception;
  
    public ForeignKey copy(Columnset newColumnSet) throws M4Exception;
}
/*
 * Historie
 * --------
 * 
 * $Log: ForeignKey.java,v $
 * Revision 1.4  2006/04/11 14:10:12  euler
 * Updated license text.
 *
 * Revision 1.3  2006/04/06 16:31:11  euler
 * Prepended license remark.
 *
 * Revision 1.2  2006/01/05 14:11:37  euler
 * Bugfixes
 *
 * Revision 1.1  2006/01/03 09:54:24  hakenjos
 * Initial version!
 *
 */
