/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4;

import edu.udo.cs.miningmart.exception.M4Exception;
import java.util.Collection;
import java.util.Iterator;

/**
 * @author Timm Euler, Daniel Hakenjos
 * @version $Id: Chain.java,v 1.4 2006/04/11 14:10:12 euler Exp $
 */
public interface Chain extends GraphicalM4Object {

	//methods from core

	// ***** This is just to work around something one could call a bug in the HCI! *****
	// public String hciBugWorkAroundGetName(); // TE: in the new HCI there won't be any bugs!!!

	/**
	 * Active getter of the steps embedded in this chain.
	 * Attention: this method returns ALL steps of ALL subchains 
	 * (including subchains of subchains, etc.)!
	 * To get only the steps that belong to this chain, ie the 
	 * top level steps, use the method <code>getTopSteps()</code>.
	 * 
	 * @return a <code>Collection</code> of <code>Step</code>s
	 */
	public Collection getAllSteps() throws M4Exception;

	/**
	 * Active getter of the top level steps of this chain.
	 * Attention: this method returns ONLY the steps of this
	 * chain; no step that belongs to subchains is returned.
	 * To get all steps, including those of subchains of subchains etc.,
	 * use the method <code>getSteps()</code>.
	 * 
	 * @return a <code>Collection</code> of <code>Step</code>s
	 */
	public Collection getTopLevelSteps() throws M4Exception;

	/**
	 * Active getter of the steps embedded in this chain. Only the
	 * steps belonging directly to this chain are returned. Steps
	 * of subchains are not included in the returned iterator. The 
	 * iterator iterates through the Steps in the right sequential 
	 * order. If there are no top level steps, NULL is returned.
	 * 
	 * @return an <code>Iterator</code> through <code>Step</code>s, or NULL.
	 */
	public Iterator getTopLevelStepsSequentialised() throws M4Exception;

	/**
	 * Returns a Collection of the Concepts that are used as input
	 * or output in any of the top level steps of this chain. Steps
	 * of subchains are not involved.
	 * 
	 * @return a Collection of Concepts
	 * @throws M4Exception
	 */
	public Collection getInvolvedConcepts() throws M4Exception;
	
	/**
	 * This method returns all steps in this chain, including steps 
	 * in subchains etc., that are a <b>direct</b> predecessor of the given Step.
	 *  
	 * @param toStep the Step whose predecessors are returned
	 * @return a Collection of Steps  
	 */
	public Collection getPredecessors(Step toStep) throws M4Exception;
	
	/**
	 * This method returns TRUE iff a Step exists in this Chain that 
	 * is a successor of a Step outside this Chain. 
	 * 
	 * @return TRUE iff this Chain has a predecessor Step outside this Chain.
	 * @throws M4Exception
	 */
	public boolean hasPredecessorOutside() throws M4Exception;

	/**
	 * This method returns TRUE iff a Step exists in this Chain that 
	 * is a predecessor of a Step outside this Chain. 
	 * 
	 * @return TRUE iff this Chain has a successor Step outside this Chain.
	 * @throws M4Exception
	 */
	public boolean hasSuccessorOutside() throws M4Exception;
	
	/**
	 * This method returns all steps in this chain, including steps 
	 * in subchains etc., that are a <b>direct</b> successor of the given Step.
	 *  
	 * @param fromStep the Step whose successors are returned
	 * @return a Collection of Steps  
	 */
	public Collection getSuccessors(Step fromStep) throws M4Exception;
	
	/**
	 * This method returns pairs of Steps. In each pair, the first Step belongs to this Chain, or one
	 * of its direct or indirect subchains, and the second to the given Chain or one of its subchains, 
	 * and there is a direct dependency (transition) between them. This method returns a
	 * Collection of Arrays of Steps. Each Array has exactly two elements. The one with index 0
	 * is a Step from this Chain, the one with index 1 is a Step from the given Chain, and a direct
	 * dependeny between the Steps exists.
	 * 
	 * @param toChain the given Chain
	 * @return a Collection of two-element Arrays of Steps
	 * @throws M4Exception
	 */
	public Collection getAllTransitionsToChain(Chain toChain) throws M4Exception;
		
	/**
	 * Active getter of the chains embedded in this chain.
	 * Attention: this method returns ALL subchains 
	 * (including subchains of subchains, etc.)!
	 * To get only the direct subchains, use the method 
	 * <code>getDirectSubChains()</code>.
	 * 
	 * @return a <code>Collection</code> of <code>Chain</code>s
	 */
	public Collection getAllChains() throws M4Exception;

	/**
	 * Active getter of the chains embedded in this chain.
	 * Attention: this method returns ONLY the direct subchains!
	 * To get all subchains, including subchains of subchains etc.,
	 * use the method <code>getAllChains()</code>.
	 * 
	 * @return a <code>Collection</code> of <code>Chain</code>s
	 */
	public Collection getDirectSubChains() throws M4Exception;
	
	/**
	 * If this chain has a parent chain, that parent chain is
	 * returned. Otherwise, this method returns NULL.
	 * 
	 * @return the direct parent chain if it exists, or <code>null</code>
	 *         if no parent chain exists.
	 * @throws M4Exception
	 */
	public Chain getParentChain() throws M4Exception;	
	
	/**
	 * This method checks if there is a dependency between any
	 * of the steps of this chain (including all steps of all
	 * direct or indirect subchains), and any of the steps of
	 * the given chain (again including all steps of all
	 * direct or indirect subchains).
	 * Attention: only the direction from this chain to the given
	 * chain is tested, not the backwards direction!
	 * 
	 * @param toChain the destination chain
	 * @return TRUE iff any of the steps in this chain has a successor 
	 * 	       in the given chain.
	 * @throws M4Exception
	 */
	public boolean dependencyExists(Chain toChain) throws M4Exception;	

	/**
	 * This method checks if there is a dependency between any
	 * of the steps of this chain (including all steps of all
	 * direct or indirect subchains), and the given step.
	 * Attention: only the direction from this chain to the given
	 * step is tested, not the backwards direction!
	 * 
	 * @param toStep the destination step
	 * @return TRUE iff any of the steps in this chain is a predecessor
	 *         of the given step.
	 * @throws M4Exception
	 */
	public boolean dependencyExists(Step toStep) throws M4Exception;
	
	/**
	 * Gets the Case.
	 * @return Returns a Case
	 */
	public Case getTheCase();

	/**
	 * Gets the description.
	 * @return Returns a String
	 */
	public String getDescription();

	/**
	 * Sets a new <code>Case</code> object and cares about
	 * back-reference integrity.
	 * 
	 * @param m4Case the <code>Case</code> to be set
	 */
	public void setTheCase(Case m4Case) throws M4Exception;

	/**
	 * Sets the description.
	 * @param description The description to set
	 */
	public void setDescription(String description);

	/**
	 * Adds a <code>Step</code> to this <code>Chain</code> and takes
	 * care about back-reference integrity. The new step is added to
	 * the top level, not to any subchains.
	 * 
	 * @param step the <code>Step</code> to be added
	 */
	public void addStep(Step step) throws M4Exception;

	/**
	 * Adds a <code>Chain</code> to this <code>Chain</code> and takes
	 * care about back-reference integrity. The given Chain will 
	 * be a direct subchain of this chain.
	 * 
	 * @param step the <code>Step</code> to be added
	 */
	public void addSubChain(Chain chain) throws M4Exception;

	/**
	 * If the given <code>Chain</code> is a direct subchain of this 
	 * <code>Chain</code>, it is resolved, ie its <code>Step</code>s 
	 * and subchains become steps and direct subchains of this chain.
	 * 
	 * @param chain the <code>Chain</code> to be resolved
	 * @throws M4Exception
	 */
	public void resolveSubChain(Chain chain) throws M4Exception;
	
	/**
	 * Creates a Chain that subsumes the given steps and chains. 
	 * It will be a direct subchain of this Chain.
	 * 
	 * @param name the name to be given to the new chain
	 * @param stepsAndChains a <code>Collection</code> of <code>Step</code>s 
	 *        and/or <code>Chain</code>s.
	 * @return the created chain
	 * @throws M4Exception
	 */
	public Chain createSubChain(String name, Collection stepsAndChains) throws M4Exception;
	
	/**
	 * Sets the parent chain of this chain to the given parentChain.
	 * 
	 * @param parentChain the new parent chain of this chain.
	 * @throws M4Exception
	 */
	public void setParentChain(Chain parentChain) throws M4Exception;
	
	/**
	 * Removes a <code>Step</code> from this <code>Chain</code> and takes
	 * care about back-reference integrity.
	 * 
	 * @param step the <code>Step</code> to be removed
	 */
	public boolean removeStep(Step step) throws M4Exception;

	//methods from old interface

	/** Copies a chain. */
	public Chain copy(Case newCase) throws M4Exception;
}
/*
 * Historie
 * --------
 * 
 * $Log: Chain.java,v $
 * Revision 1.4  2006/04/11 14:10:12  euler
 * Updated license text.
 *
 * Revision 1.3  2006/04/06 16:31:12  euler
 * Prepended license remark.
 *
 * Revision 1.2  2006/01/24 14:03:49  euler
 * Added display of concepts involved in a chain.
 *
 * Revision 1.1  2006/01/03 09:54:24  hakenjos
 * Initial version!
 *
 */
