/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;

import edu.udo.cs.miningmart.exception.M4Exception;

/**
 * @author Timm Euler, Daniel Hakenjos
 * @version $Id: Case.java,v 1.3 2006/04/11 14:10:12 euler Exp $
 */
public interface Case extends M4Data {

	//methods from core
	
	/**
	 * This method adds dependencies between <code>Step</code>s and brings the
	 * <code>Step</code>s in a sequential order again, if necessary and if they
	 * have been in sequential order before adding the dependency.
	 * 
	 * @param before a <code>Step</code> part of this <code>Case</code>
 	 * @param after a <code>Step</code> part of this <code>Case</code> and depending
 	 *              on <code>Step</code> <code>before</code>.
	 */
	public void addStepDependency(Step before, Step after)
		throws M4Exception;

	/*
	 * This method adds dependencies between <code>Step</code>s and brings the
	 * <code>Step</code>s in a sequential order again, if necessary and if they
	 * have been in sequential order before adding the dependency.
	 * 
	 * @param before a <code>Step</code> part of this <code>Case</code>
 	 * @param after a <code>Step</code> part of this <code>Case</code> and depending
 	 *              on <code>Step</code> <code>before</code>.
	 *
	public void addStepToStepDependency(Step before, Step after)
		throws M4Exception;
		
	/*
	 * This method adds dependencies between a <code>Step</code> and a <code>Chain</code> and brings them
	 * in a sequential order again, if necessary and if they
	 * have been in sequential order before adding the dependency.
	 * 
	 * @param before a <code>Step</code> part of this <code>Case</code>
 	 * @param after a <code>Chain</code> part of this <code>Case</code> and depending
 	 *              on <code>Step</code> <code>before</code>.
	 *
	public void addStepToChainDependency(Step before, Chain after)
		throws M4Exception;

	/*
	 * This method adds dependencies between a <code>Chain</code> and a <code>Step</code> and brings them
	 * in a sequential order again, if necessary and if they
	 * have been in sequential order before adding the dependency.
	 * 
	 * @param before a <code>Chain</code> part of this <code>Case</code>
 	 * @param after a <code>Step</code> part of this <code>Case</code> and depending
 	 *              on <code>Chain</code> <code>before</code>.
	 *
	public void addChainToStepDependency(Chain before, Step after)
		throws M4Exception;

	/*
	 * This method adds dependencies between <code>Chain</code>s and brings them
	 * in a sequential order again, if necessary and if they
	 * have been in sequential order before adding the dependency.
	 * 
	 * @param before a <code>Chain</code> part of this <code>Case</code>
 	 * @param after a <code>Chain</code> part of this <code>Case</code> and depending
 	 *              on <code>Chain</code> <code>before</code>.
	 *
	public void addChainToChainDependency(Chain before, Chain after)
		throws M4Exception;
    */

	/**
	 * Checks for a <b>direct</b> (explicitly added) dependency between the two
	 * <code>Step</code>s.
	 * 
	 * @param stepBefore a <code>Step</code> part of this <code>Case</code>
 	 * @param stepAfter another <code>Step</code> part of this <code>Case</code>
 	 * @return <code>true</code> if the second <code>Step</code> directly depends on
 	 * 		   the first one. If one of the <code>Step</code>s is not part of the
 	 * 		   <code>Case</code> then <code>false</code> is returned.
	 */
	public boolean containsDependency(Step before, Step after)
		throws M4Exception;

	/*
	 * Checks for a <b>direct</b> (explicitly added) dependency between a
	 * <code>Step</code> and a <code>Chain</code>.
	 * 
	 * @param stepBefore a <code>Step</code> part of this <code>Case</code>
 	 * @param chainAfter a <code>Chain</code> part of this <code>Case</code>
 	 * @return <code>true</code> if the <code>Chain</code> directly depends on
 	 * 		   the <code>Step</code>. If the <code>Step</code> or the <code>Chain</code> is not part of the
 	 * 		   <code>Case</code> then <code>false</code> is returned.
	 *
	public boolean containsDependency(Step before, Chain after)
		throws M4Exception;

	/*
	 * Checks for a <b>direct</b> (explicitly added) dependency between a
	 * <code>Chain</code> and a <code>Step</code>.
	 * 
	 * @param chainBefore a <code>Chain</code> part of this <code>Case</code>
 	 * @param stepAfter a <code>Step</code> part of this <code>Case</code>
 	 * @return <code>true</code> if the <code>Chain</code> directly depends on
 	 * 		   the <code>Chain</code>. If the <code>Step</code> or the <code>Chain</code> is not part of the
 	 * 		   <code>Case</code> then <code>false</code> is returned.
	 *
	public boolean containsDependency(Chain before, Step after)
		throws M4Exception;

	/*
	 * Checks for a <b>direct</b> (explicitly added) dependency between the two
	 * <code>Chain</code>s.
	 * 
	 * @param chainBefore a <code>Chain</code> part of this <code>Case</code>
 	 * @param chainAfter another <code>Chain</code> part of this <code>Case</code>
 	 * @return <code>true</code> if the second <code>Chain</code> directly depends on
 	 * 		   the first one. If one of the <code>Chain</code>s is not part of the
 	 * 		   <code>Case</code> then <code>false</code> is returned.
	 *
	public boolean containsDependency(Chain before, Chain after)
		throws M4Exception;
	 */
	
	/**
	 * A dependency (or edge in graph terminology) between the two <code>Step</code>s
	 * is removed.
	 * 
	 * @param before a <code>Step</code> part of this <code>Case</code>
 	 * @param after  a <code>Step</code> part of this <code>Case</code>, 
 	 *               and depending on <code>Step</code> <code>before</code>
	 * @return <code>true</code> iff the dependency was found and could be removed
	 */
	public boolean removeStepDependency(Step before, Step after)
		throws M4Exception;

	/*
	 * A dependency (or edge in graph terminology) between the two <code>Step</code>s
	 * is removed.
	 * 
	 * @param before a <code>Step</code> part of this <code>Case</code>
 	 * @param after  a <code>Step</code> part of this <code>Case</code>, 
 	 *               and depending on <code>Step</code> <code>before</code>
	 * @return <code>true</code> iff the dependency was found and could be removed
	 *
	public boolean removeStepToStepDependency(Step before, Step after)
		throws M4Exception;

	/*
	 * A dependency (or edge in graph terminology) between a<code>Step</code> and a <code>Chain</code>
	 * is removed.
	 * 
	 * @param before a <code>Step</code> part of this <code>Case</code>
 	 * @param after  a <code>Chain</code> part of this <code>Case</code>, 
 	 *               and depending on <code>Step</code> <code>before</code>
	 * @return <code>true</code> iff the dependency was found and could be removed
	 *
	public boolean removeStepToChainDependency(Step before, Chain after)
		throws M4Exception;

	/*
	 * A dependency (or edge in graph terminology) between a<code>Chain</code> and a <code>Step</code>
	 * is removed.
	 * 
	 * @param before a <code>Chain</code> part of this <code>Case</code>
 	 * @param after  a <code>Step</code> part of this <code>Case</code>, 
 	 *               and depending on <code>Chain</code> <code>before</code>
	 * @return <code>true</code> iff the dependency was found and could be removed
	 *
	public boolean removeChainToStepDependency(Chain before, Step after)
		throws M4Exception;



	/*
	 * A dependency (or edge in graph terminology) between the two <code>Chain</code>s
	 * is removed.
	 * 
	 * @param before a <code>Chain</code> part of this <code>Case</code>
 	 * @param after  a <code>Chain</code> part of this <code>Case</code>, 
 	 *               and depending on <code>Chain</code> <code>before</code>
	 * @return <code>true</code> iff the dependency was found and could be removed
	 *
	public boolean removeChainToChainDependency(Chain before, Chain after)
		throws M4Exception;
	*/


	/**
	 * Set the Case mode. Allowed constants are given by the public static
	 * variables TESTMODE and FINALMODE of this class.
	 * 
	 * @param newMode The new case mode. Use the public static variables
	 *         TESTMODE and FINALMODE of this class.
	 */
	public void setTheMode(String newMode) throws M4Exception;
	
	/**
	 * Get the Case mode.
	 * 
	 * @return A String with the value of one of the public static fields 
	 *         TESTMODE and FINALMODE of this class.
	 */
	public String getTheMode();

	
	/**
	 * Gets the step <i>i</i> in sequential order.
	 * 
	 * @return Returns a step
	 */
	public Step getStepIdNo(int i) throws M4Exception;

	/**
	 * @return the number of steps of this case.
	 */
	public int getNumberOfSteps() throws M4Exception; 
	
	/**
	 * Gets the predecessor step, given the calculated sequential order.
	 * 
	 * @param step The step for which the predecessor is wanted.
	 * @return the predecessor step
	 */
	public Step getPredecessorIdOf(Step step) throws M4Exception;
	/**
	 * Gets the successor step, given the calculated sequential order.
	 * 
	 * @param step The step for which the successor is wanted.
	 * @return the successor step
	 */
	public Step getSuccessorOf(Step step) throws M4Exception;

    /**
     * @return the index of the <i>Step</i> with the given Id, if part of this case.
     *         The result depends of the chosen (but fixed) sequential order of the
     *         <code>Sequentializer</code>!
     */
	public int indexOfStep(Step step) throws M4Exception;

	/**
 	 * @return an <i>Iterator</i> providing the <i>Step</i>s of
 	 *         this <i>Case</i>.
     *         The order depends on the <code>Sequentializer</code>.
 	 */
	public Iterator getStepIterator() throws M4Exception;

	/**
	 * @return an <i>Iterator</i> providing the <i>Step</i>s in reverse order.
     *         The order depends on the <code>Sequentializer</code>!
	 */
	public Iterator getReverseIterator() throws M4Exception;

	/**
	 * Gets a <code>Collection</code> of the stepIds that are dependent on
	 * the given step, ie must be compiled after the given step.
	 * 
	 * @param startStepId The step for which the dependent steps are wanted.
	 * @return The dependent steps as a Collection.
	 */
	public Collection getDependentStepsFor(Step step) 
		throws M4Exception;	

	/**
	 * This method returns a Collection of all Steps on which the
	 * given Step is dependent. That means, all Steps are returned 
	 * that must be compiled before the given Step can be compiled.
	 * By the parameter ignoreCompiledStatusOfSteps, one can specify
	 * whether the method considers only the graph structure of M4 Step
	 * dependencies, and ignores whether some of the Steps to be returned
	 * have already been compiled (and thus would not have to be
	 * compiled again) - or whether this compilation status of Steps
	 * should be taken into account. In the latter case, the search 
	 * for predecessors of the given Steps ends, on every path that
	 * leads to the given Step, at the last Step that
	 * is already compiled. 
	 * 
	 * @param step the Step whose preceding graph is returned
	 * @param ignoreCompiledStatusOfSteps if TRUE, *all* Steps that lead
	 * to the given Step are returned; if FALSE, only those starting
	 * at the last compiled Step on each path that leads to the given Step
	 * are returned.
	 * @return a Collection of all Steps that must, in principle, be
	 *     compiled before the given Step can be compiled
	 * @throws M4Exception
	 */
	public Collection getStepsToCompileBefore(edu.udo.cs.miningmart.m4.Step step,
			                                  boolean ignoreCompiledStatusOfSteps)
	       throws M4Exception;
	
	/**
	 * Gets a <code>Collection</code> of the steps that are directly
	 * dependent on the given step. This is useful for the HCI to just
	 * visualize the explicitly stored dependencies.
	 * 
	 * @param step The step for which the dependent steps are wanted.
	 * @return The dependent steps as a Collection or <code>null</code>,
	 * if the specified <code>Step</code> is not found.
	 */
	public Collection getDirectlyDependentSteps(Step step)
		throws M4Exception;

	/**
	 * Gets the names of all concepts.
	 * @return The names (String).
	 * @throws M4Exception
	 */
	public Collection getAllConceptNames() throws M4Exception;

	/**
	 * Active getter of the Concepts of this Case.
	 *
	 * @return a <code>Collection</code> of <code>Concept</code>s
	 */
	public Collection getConcepts() throws M4Exception;

	/**
	 * Returns a collection of all concepts of type DB in this Case.
	 * 
	 * @return a <code>Collection</code> of <code>Concept</code>s
	 * @throws M4Exception
	 */
	public Collection getInputDataModel() throws M4Exception;
	
	/**
	 * Adds a new Concept to this case.
	 * @param con the new <code>Concept</code>
	 */
	public void addConcept(Concept con) throws M4Exception;

	/**
	 * Removes a Concept from this case.
	 * @param con the <code>Concept</code> to be removed
	 * @return <code>true</code> if <code>Concept</code> was found and could
	 * be removed
	 */
	public boolean removeConcept(Concept con) throws M4Exception;

	/**
	 * Active getter of the Chains of this Case. 
	 * Attention: It returns ALL chains of this Case, 
	 * including subchains of subchains etc.!
	 * Compare to method <code>getTopLevelChains</code>.
	 *
	 * @return a <code>Collection</code> of <code>Chain</code>s
	 */
	public Collection getAllChains() throws M4Exception;
	
	/**
	 * Active getter of the Chains of this Case. 
	 * Attention: It returns ONLY the top-level chains of this Case, 
	 * that is, the Chains that do not have a parent chain.
	 * Compare to method <code>getAllChains</code>.
	 *
	 * @return a <code>Collection</code> of <code>Chain</code>s
	 */
	public Collection getTopLevelChains() throws M4Exception;
	
	/**
	 * Retrieve a Chain by its name.
	 * 
	 * @param name the Name
	 * @throws M4Exception
	 */
	public Chain getChainByName(String name) throws M4Exception;
	
	/**
	 * Remove a Chain by its name.
	 * 
	 * @param name the Name
	 * @throws M4Exception
	 */
	public void removeChain(String name) throws M4Exception;

	/**
	 * Adds a new Chain to this case. It does not have to be a top level
	 * chain (ie a chain without a parent chain).
	 * 
	 * @param chain the new <code>Chain</code>
	 */
	public void addChain(Chain chain) throws M4Exception;

	/**
	 * Removes a Chain from this case.
	 * @param  chain the <code>Chain</code> to be removed
	 * @return <code>true</code> if <code>Chain</code> was found and could
	 *         be removed
	 */
	public boolean removeChain(Chain chain) throws M4Exception;


	/**
	 * This method finds a <code>Step</code> related to this <code>Case</code>
	 * by the <code>Step</code>'s M4 name
	 * @param name the name of the <code>Step</code> object
	 * @return the <code>Step</code> or <code>null</code>
	 */
	public Step getStepByName(String name) throws M4Exception;

	/** 
	 * Active getter
	 * 
	 * @return a <code>Collection</code> of all <code>Step</code>s of this <code>Case</code>
	 */
	public Vector getTheSteps() throws M4Exception;

	/**
	 * Adds a <code>Step</code> to this <code>Case</code>.
	 * @param step the <code>Step</code> to be added
	 * */
	public void addStep(Step step) throws M4Exception;

	/**
	 * Removes a <code>Step</code> and all of its dependencies from this
	 * <code>Case</code>.
	 * @param step the <code>Step</code> to be added
	 */
	public boolean removeStep(Step step) throws M4Exception;


	//methods from old interface
	
	/** Constant that can be used to set the Case Mode to TEST */
  public static final String TEST_MODE = "TEST";

	/** Constant that can be used to set the Case Mode to DESIGN */
  public static final String DESIGN_MODE = "DESIGN";

	/** Constant that can be used to set the Case Mode to FINAL */
  public static final String FINAL_MODE = "FINAL";

  /* Attribute methods */

  // Concept getPopulation(); // TE: not needed

  // void setPopulation(Concept population) throws StorageException; // TE: not needed

  // Concept getOutput(); // TE: not needed

  // void setOutput(Concept output) throws StorageException; // TE: not needed

  public Case copy() throws M4Exception;

  /*
   * This method may be used by the client to manage a higher form
   * of validity for an object. At this point it has not yet been
   * defined however what 'functionally valid' means! Currently this information
   * is also not saved in the M4 Meta Data Schema.
   *
  boolean isFunctionallyValid(); // TE: not needed for Case
  */

  /*
   * This method may be used by the client to manage a higher form
   * of validity for an object. At this point it has not yet been
   * defined however what 'functionally valid' means! Currently this information is also not saved in the M4 Meta Data Schema.
   *
  void setFunctionallyValid(boolean functionallyValid); // TE: not needed for Case
   */

  /*
   * This method returns information about the relational validity of the object. The relational validity is managed at the
   * database level by triggers. The returned value denotes if the object is valid on the relational level or not.
   *
  boolean isRelationallyValid(); // TE: not needed for Case
   */

  /* Association methods */

  /**
   * Creates an empty Step belonging to this Case. The name of the
   * Step should be unique within the Case.
   * @throws CreateException when an error occurs during creation of the object.
   * @throws NameExistsException when the provided name already exists.
   */
  Step createStep(String name) throws M4Exception;

  /**
   * Returns a Collection with all Step names in the current Case.
   */
  Collection getAllStepNames() throws M4Exception;

  /**
   * The specified Step will be removed including its Parameters. Concepts
   * and BaseAttributes of types BASE and DB however will not be removed.
   */
  void removeStep(String name) throws M4Exception;

  /**
   * All Steps of a Case will be removed including their Parameters. Concepts
   * and BaseAttributes of types BASE and DB however will not be removed.
   */
  void removeAllSteps() throws M4Exception;

  // void addCaseInput(M4Object object); // TE: not needed

  // Collection getAllCaseInputs(); // TE: not needed

  /*
   * The specified CaseInput will be removed as CaseInput. The object itself
   * will still exist in the Case.
   */
  // void removeCaseInput(M4Object object) throws StorageException; // TE: not needed

  /*
   * All CaseInputs of this Case will be removed.
   * The objects themselves will still exist in the Case however.
   */
  // void removeAllCaseInputs() throws StorageException; // TE: not needed

  // void addCaseAttribute(BaseAttribute attribute); // TE: not needed

  // Collection getAllCaseAttributes(); // TE: not needed

  /*
   * The specified CaseAttribute will be removed. The BaseAttribute will
   * still exist in the Case.
   */
  // void removeCaseAttribute(BaseAttribute attribute) throws StorageException; // TE: not needed

  /*
   * All CaseAttribute records referring to this Case will be removed.
   * The BaseAttributes themselves will still exist in the Case however.
   */
  // void removeAllCaseAttributes() throws StorageException; // TE: not needed

  /**
   * Creates an empty Chain belonging to this Case. The name of the
   * Chain should be unique within the Case.
   *
   * @param name the name to be given to the new chain
   * @return the created chain
   * @throws M4Exception
   */
  Chain createChain(String name) throws M4Exception;
  
  /**
   * Creates a Chain that subsumes the given steps and chains. It
   * will not be given any parent chain, so it will be a top level
   * chain in this Case.
   * 
   * @param name the name to be given to the new chain
   * @param stepsAndChains a <code>Collection</code> of <code>Step</code>s 
   *        and/or <code>Chain</code>s.
   * @return the created chain
   * @throws M4Exception
   */
  Chain createChain(String name, Collection stepsAndChains) throws M4Exception;

  /**
   * The given chain will be resolved if it is a top level chain
   * and if it does not have any Steps, only subchains. Otherwise
   * an M4Exception is thrown. The subchains of the given chain will be 
   * new top-level chains of this Case.
   */
  public void resolveChain(Chain toBeResolved) throws M4Exception;
	
  /**
   * Creates an empty Concept belonging to this Case. The name of the
   * Concept must be unique within the Case.
   * 
   * @param name the name for the concept to be created
   * @param type the type of the concept (see Concept.TYPE_...)
   * @throws M4Exception
   */
  Concept createConcept(String name, String type) throws M4Exception;

  /**
   * Creates a Concept that belongs to this Case and that directly
   * models a database table or view with the given name. The concept will 
   * have exactly the given name. For each column of the table/view a 
   * BaseAttribute is created and attached to the Concept. The given name 
   * must be a valid name of a database table or view in the current 
   * business data schema, otherwise an exception is thrown.
   * 
   * @param tableName the name of a table or view in the business schema
   * @throws M4Exception
   */
  Concept createConceptFromTable(String tableName) throws M4Exception;

  /**
   * Creates a Concept that belongs to this Case and is connected to the
   * given Columnset. The concept directly reflects the Columnset, that is,
   * its name and the names of the BaseAttributes are taken from the Columnset
   * and its Columns.
   * 
   * @param cs the given Columnset
   * @return the new Concept
   * @throws M4Exception
   */
  public Concept createConceptFromColumnset(Columnset cs) throws M4Exception;
  
  /**
   * Creates a one-to-many relation. The name must be valid for this Case. 
   * There must be equally many Features in the from/to key attribute 
   * collections.
   * 
   * @param relName a valid name for a new Relation in this Case
   * @param fromConceptKeyAttribs a Collection of the Features that make up the key
   * attributes in the FromConcept
   * @param toConceptKeyAttribs a Collection of the Features that make up the key
   * attributes in the ToConcept
   * @return the newly created Relation
   * @throws M4Exception
   */
  public edu.udo.cs.miningmart.m4.Relation createOneToManyRelation( String relName, 
																 Collection fromConceptKeyAttribs,
																 Collection toConceptKeyAttribs)
  throws M4Exception;
  
  /**
   * Creates a many-to-many relation. The name must be valid for this Case. 
   * There must be equally many Features in the from/to key attribute 
   * collections as there are names in the from/to cross link name collections.
   * 
   * @param relName a valid name for a new Relation in this Case
   * @param fromConceptKeyAttribs a Collection of the Features that make up the key
   * attributes in the FromConcept
   * @param toConceptKeyAttribs a Collection of the Features that make up the key
   * attributes in the ToConcept
   * @param crossTableName the name of the table in the business data schema that 
   * serves as the cross table
   * @param crossLinkToFromConceptNames a Collection with the names of the cross table 
   * columns that provide the foreign key to the primary key of the FromConcept
   * @param crossLinkToToConceptNames a Collection with the names of the cross table
   * columns that provide the foreign key to the primary key of the ToConcept
   * @return the newly created Relation
   * @throws M4Exception
   */
  public edu.udo.cs.miningmart.m4.Relation createManyToManyRelation( String relName, 
																  Collection fromConceptKeyAttribs,
																  Collection toConceptKeyAttribs,
															  	  String crossTableName,
																  Collection crossLinkToFromConceptNames,
																  Collection crossLinkToToConceptNames)
  throws M4Exception;
  
  /**
   * Creates many-to-many relations with the given table name as cross table.
   * This only works if the foreign key information is stored in the database.
   * 
   * @param crossTableName
   * @return
   * @throws M4Exception
   */
  public Collection createManyToManyRelationsFromCrossTable(String crossTableName) 
  throws M4Exception;

  /**
   * This method creates a Columnset and Concept for the given table/view name,
   * and also creates a Columnset and Concept for EVERY table that is referenced from 
   * the first one by a foreign key reference, and also creates an M4 Relation for 
   * those references. Concepts and Columnsets for the other tables/views are
   * only created if no such object with the same name exists yet in the M4 cache.
   * At the moment only one-to-many relations are created!
   *  
   * @param tableName the given table name
   * @return the first Concept, from which the referenced Concepts can be found by
   * following the M4 Relations.
   */
  public edu.udo.cs.miningmart.m4.Concept createConceptAndRelationsFromTables(String tableName) 
  throws M4Exception;
	
  Concept getConcept(String conceptName) throws M4Exception;

  /**
   * Write all M4 data pertaining to this case through to the
   * database.
   * 
   * @throws M4Exception
   */
  void store() throws M4Exception;
  
  	/**
	 * Returns all Relations that this Case has.
	 * 
	 * @return a Collection of Relations
	 * @throws M4Exception
	 */
	public Collection getAllRelations() throws M4Exception;

	/**
	 * Returns a map that maps operators that occur in this
	 * case to a collection of the steps in which they occur.
	 *  
	 * @return a map from operators to collections of steps
	 * @throws M4Exception
	 */
	public Map getOperatorToStepMap() throws M4Exception;
  
   /*
    * Return all available Values.
    * @return A java.util.Collection containing all Values.
    */
  // java.util.Collection getAllValues(); // TE: what do you need this for? 

   /*
    * Returns all available Value Names not only of the current Case
    * but from all Cases. (A Value is not unique for a Case.)
    * @return A java.util.Collection containing all Value Names.
    */
  // java.util.Collection getAllValueNames(); // TE: what do you need this for? 

   /*
    * Removes all Values of this Case. The objects are removed from
    * the M4 Schema.
    */
  // public void removeAllValues() throws M4Exception; // TE: what do you need this for? 	
}
/*
 * Historie
 * --------
 * 
 * $Log: Case.java,v $
 * Revision 1.3  2006/04/11 14:10:12  euler
 * Updated license text.
 *
 * Revision 1.2  2006/04/06 16:31:12  euler
 * Prepended license remark.
 *
 * Revision 1.1  2006/01/03 09:54:24  hakenjos
 * Initial version!
 *
 */
