/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.stepsettings;

import java.util.Iterator;

import javax.swing.event.TableModelEvent;
import javax.swing.table.AbstractTableModel;

import org.musoft.limo.application.Resource;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.exception.MiningMartException;
import edu.udo.cs.miningmart.m4.M4Interface;
import edu.udo.cs.miningmart.m4.OpParam;
import edu.udo.cs.miningmart.m4.Operator;
import edu.udo.cs.miningmart.m4.Step;
import edu.udo.cs.miningmart.m4.utils.Print;

/**
 * A Model for the <code>StepParameterTable</code>.
 * @author Daniel Hakenjos
 * @version $Id: StepParameterModel.java,v 1.3 2006/04/11 14:10:13 euler Exp $
 */
public class StepParameterModel extends AbstractTableModel{

	private Step step;
	private Operator operator;
	
	private boolean editable;
	
	private String[] paramnames;
	
	private int loopnumber;
	private int nrOfLoops;
	
	private int oldNrOfLoops;
	
        /**
         *Constructs a new <code>StepParameterModel</code>.
         */
	public StepParameterModel(Step step, boolean editable){
		this.step=step;
		this.operator=step.getTheOperator();
		this.editable=editable;
		loopnumber=1;
		nrOfLoops=step.getLoopCount();
		if (nrOfLoops==0)
			nrOfLoops=1;
		oldNrOfLoops=nrOfLoops;
		initParameterNames();
	}
	
	private void initParameterNames(){
		paramnames=new String[getRowCount()];

		try{
			Iterator iter=step.getTheOperator().getOpParamsIterator();
	
			int p=0;
			OpParam opparam;
			while ((iter.hasNext())&&(p<paramnames.length)){
				opparam=(OpParam) iter.next();
				paramnames[p]=opparam.getName();
				p++;
			}
			if (p!=paramnames.length){
				System.out.println("ParameterNames not all catched: "+p);
			}
		}catch(MiningMartException error){
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
		}
	}
	
    /**
     *Sets the loopnumber.
     */
	public void setLoopNumber(int loopnumber){
		this.loopnumber=loopnumber;
		fireTableDataChanged();
		this.fireTableChanged(new TableModelEvent(this,0,getRowCount()));
	}
	
	/**
	 * Sets the number of loops.
	 */
	public void setNrOfLoops(int nrOfLoops){
		if (nrOfLoops==0){
			nrOfLoops=1;
		}
		this.nrOfLoops=nrOfLoops;
	}
	
	/**
	 * Gets the number of loops.
	 */
	public int getNrOfLoops(){
		return nrOfLoops;
	}

	/**
	 * Gets the old number of loops.
	 */	
	public int getOldNrOfLoops(){
		return oldNrOfLoops;
	}

	/**
	 * Sets the old number of loops.
	 */	
	public void setOldNrOfLoops(int oldNrOfLoops){
		this.oldNrOfLoops=oldNrOfLoops;
	}
	
	/**
	 * Only two columns. One for the parameter-name and one for the value(s).
	 * @see javax.swing.table.TableModel#getColumnCount()
	 */
	public int getColumnCount() {
		return 2;
	}

	/**
	 * Gets the row count.
	 * @return size of the OpParams-Collection.
	 * @see javax.swing.table.TableModel#getRowCount()
	 */
	public int getRowCount() {
		int rows=0;
		try{
			if (step != null && step.getTheOperator() != null) {
				rows=step.getTheOperator().getOpParams().size();
			}
			else rows = 0;
		}catch(MiningMartException error){
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
			rows=0;
		}
		return rows;
	}

	/**
	 * Gets the Value.
	 * Its a ParameterArray.
	 * @see javax.swing.table.TableModel#getValueAt(int, int)
	 */
	public Object getValueAt(int rowIndex, int colIndex) {
		if (colIndex==0){
			return getParameterName(rowIndex);
		}
		return new String("");
	}

	/**
	 * Gets the name of the parameter with the specified index.
	 */	
	public String getParameterName(int rowIndex){
		if (paramnames[rowIndex]!=null){
			return paramnames[rowIndex];
		}
		return "";
	}
		

	/**
	 * @see javax.swing.table.TableModel#getColumnName(int)
	 */
	public String getColumnName(int columnIndex) {
		if (columnIndex==0){
			return Resource.getString("STEPSETTINGS_TABLE_PARAMETER");
		}
		if (columnIndex==1){
			return Resource.getString("STEPSETTINGS_TABLE_VALUE");
		}
		
		return "";
	}

	/**
	 * @see javax.swing.table.TableModel#isCellEditable(int, int)
	 */
	public boolean isCellEditable(int rowIndex, int columnIndex) {

		//die erste Spalte der Tabelle ist nicht editierbar
		if (columnIndex==0)
			return false;

		return true;		

/*		//Nur im EDITOR-MODE kann etwas editiert werden.
		if (editable){
			//wenn der Operator nicht loopable ist, so muessen alle Parameter editierbar sein.
			if (!step.getTheOperator().isLoopable()){
				return true;
			}else{
				if (loopnumber>1){
					try{
						if (getOpParam(rowIndex).isLoopable()){
							return true;
						}
					}catch(M4Exception error){
						M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
					}
					return false;
				}
				return true;
			}
		}
		
		return false;*/

	}

	/**
	 * @see javax.swing.table.TableModel#setValueAt(Object, int, int)
	 */
	public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
		super.setValueAt(aValue, rowIndex, columnIndex);
	}

	/**
	 * Gets the Type of the Parameter in the specified row.
	 * @return -1 if errors occured.
	 * @see edu.udo.cs.miningmart.m4.core.Parameter#TYPE_VALUE
	 * @see edu.udo.cs.miningmart.m4.core.Parameter#TYPE_CONCEPT
	 * @see edu.udo.cs.miningmart.m4.core.Parameter#TYPE_RELATION
	 * @see edu.udo.cs.miningmart.m4.core.Parameter#TYPE_BASEATTRIBUTE
	 * @see edu.udo.cs.miningmart.m4.core.Parameter#TYPE_MULTICOLUMNFEATURE
	 * @see edu.udo.cs.miningmart.m4.core.Parameter#TYPE_FEATURE
	 */	
	public short getType(int row){
		try{
			OpParam opparam=(OpParam) step.getTheOperator().getOpParams().toArray()[row];
			return opparam.getType();
		}catch(M4Exception error){
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
		}
		return -1;
	}
	
	/**
	 * Is the parameter in this row an input-parameter
	 */
	public boolean isInput(int row){
		try{
			OpParam opparam=(OpParam) step.getTheOperator().getOpParams().toArray()[row];
			return opparam.isInput();
		}catch(M4Exception error){
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
		}
		return false;
	}
	
	/**
	 * Gets the <code>OpParam</code> in this row.
	 */
	public OpParam getOpParam(int row){
		try{
			OpParam opparam=(OpParam) step.getTheOperator().getOpParams().toArray()[row];
			return opparam;
		}catch(M4Exception error){
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
		}
		return null;
	}
	
        /**
         *Gets the <code>OpParam</code> by the specified name.
         */
	public OpParam getOpParamByName(String opParamName){
		for (int n=0;n<paramnames.length;n++){
			if (paramnames[n].equals(opParamName)){
				return getOpParam(n);
			}
		}
		return null;
	}
	
        /**
         *Gets the row of the specified <code>OpParam</code>-name.
         */
	public int getRowOfOpParamByName(String opParamName){
		for (int n=0;n<paramnames.length;n++){
			if (paramnames[n].equals(opParamName)){
				return n;
			}
		}
		return -1;
	}

}
/*
$Log: StepParameterModel.java,v $
Revision 1.3  2006/04/11 14:10:13  euler
Updated license text.

Revision 1.2  2006/04/06 16:31:12  euler
Prepended license remark.

Revision 1.1  2006/01/03 09:54:14  hakenjos
Initial version!

*/
