/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.stepsettings;

import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeListener;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingConstants;

import org.musoft.limo.application.Application;
import org.musoft.limo.application.Resource;

import edu.udo.cs.miningmart.gui.application.IteratorList;
import edu.udo.cs.miningmart.gui.application.MiningMartApplication;
import edu.udo.cs.miningmart.m4.M4Interface;
import edu.udo.cs.miningmart.m4.Step;
import edu.udo.cs.miningmart.m4.utils.Print;

/**
 *A <code>JPanel</code> for changes the number of loops and changing the loopnumber.
 * @author Daniel Hakenjos
 * @version $Id: StepLoopPanel.java,v 1.3 2006/04/11 14:10:13 euler Exp $
 */
public class StepLoopPanel extends JPanel implements ActionListener {

	private MiningMartApplication app;

	private Step step;
	
	private int oldnumber=0,number=0;
	
	private JButton change;
	
	private JTextField loopsfield;

	private	IteratorList iterlist;
	
	private Vector looplistener;

    /**
     *Constructs new <code>StepLoopPanel</code>.
     */
	public StepLoopPanel(MiningMartApplication app, Step step) {

		this.app=app;
		this.step=step;
		looplistener=new Vector();
		initComponents();
	}
	
	/**
	 * Adds a <code>NrOfLoopsListener</code>.
	 */
	public void addNrOfLoopsListener(NrOfLoopsListener listener){
		looplistener.add(listener);
	}
	
	/**
	 * Removes the specified <code>NrOfLoopsListener</code>.
	 */
	public void removeNrOfLoopsListener(NrOfLoopsListener listener){
		looplistener.remove(listener);
	}
	
	/**
	 * Removes all <code>NrOfLoopsListener</code>.
	 */
	public void reamoveAllNrOfLoopsListener(){
		looplistener.removeAllElements();
	}
	
	/**
	 * Counts  <code>NrOfLoopsListener</code>s.
	 */
	public int getNrOfLoopsListenerCount(){
		return looplistener.size();
	}
	
	/**
	 * Gets the <code>NrOfLoopsListener</code> for the specified index.
	 */
	public NrOfLoopsListener getNrOfLoopsListener(int index){
		return (NrOfLoopsListener) looplistener.get(index);
	}

	/**
	 * Informs all <code>NrOfLoopsListener</code>s.
	 */	
	public void fireNrOfLoopsChanged(){
		NrOfLoopsListener listener;
		for(int i=0;i<looplistener.size();i++){
			listener=getNrOfLoopsListener(i);
			listener.numberOfLoopsChanged(getNumberOfLoops(false));
		}
	}
	

	private void initComponents(){
		this.setLayout(new FlowLayout());
//		this.setBorder(BorderFactory.createLineBorder(Color.black));

		JLabel label1=new JLabel(Resource.getString("NUMBER_OF_LOOPS"));
		label1.setHorizontalAlignment(SwingConstants.LEFT);
		label1.setHorizontalTextPosition(SwingConstants.LEFT);
		this.add(label1);

		loopsfield=new JTextField("0",2);
		loopsfield.setText(Integer.toString(step.getLoopCount()));
//		System.out.println("LoopCount: "+step.getLoopCount());
		loopsfield.setHorizontalAlignment(SwingConstants.RIGHT);
		if (app.getViewMode()!=Application.EDITOR){
			loopsfield.setEnabled(false);
		}
		this.add(loopsfield);


		change=new JButton(Resource.getString("CHANGE_LOOP_NUMBER"));
		if (app.getViewMode()==Application.EDITOR){
			change.addActionListener(this);
		}else{
			change.setEnabled(false);
		}
		this.add(change);

		JLabel label2=new JLabel(Resource.getString("SHOW_LOOP_NUMBER"));
		this.add(label2);

		iterlist=new IteratorList();
		setData(step.getLoopCount());
		this.add(iterlist);

		if (app.isReadOnly()) {
			change.setEnabled(false);
			loopsfield.setEnabled(false);
		} else {
			change.setEnabled(true);
			loopsfield.setEnabled(true);
		}
	}
	
    /**
     *Gets the <code>IteratorList</code>.
     */
	public IteratorList getIteratorList(){
		return iterlist;
	}

    /**
     *Sets the number of the loops.
     */
	public void setNumberOfLoops(int nrOfLoops) {
		loopsfield.setText(nrOfLoops + "");
		setData(nrOfLoops);
		if (nrOfLoops>=0){
//			app.getModel().setDirty(true);
//			step.setLoopCount(nrOfLoops);
			fireNrOfLoopsChanged();
		}
	}

    /**
     *Gets the number of loops.
     * @param error if true and errors occures then show a dialog.
     */
	public int getNumberOfLoops(boolean error) {
		int nr = 0;

		try {
			nr = Integer.parseInt(loopsfield.getText());
		} catch (NumberFormatException e) {
			if (error){
				JOptionPane.showMessageDialog(
					app,
					Resource.getString("STEPSETTINGS_LOOPNUMBER_ERROR"),
					Resource.getString("DIALOG_ERROR_TITLE"),
					JOptionPane.ERROR_MESSAGE);
			}else{
				M4Interface.print.doPrint(Print.ERROR,e.getMessage(),e);
			}
		}
		return nr;
	}

        /**
         *Gets the current loopnumber.
         */
	public int getCurrentLoop() {
		int erg = 0;
		if (getNumberOfLoops(false) > 0) {
			String value = iterlist.getSelected();
			try {
				erg = Integer.parseInt(value);
			} catch (Exception e) {
				//this exception should not occure because the values of the spinner are integers
				M4Interface.print.doPrint(Print.ERROR,e.getMessage(),e);
			}
		}
		return erg;
	}

        /**
         *Sets the data to the <code>IteratorList</code>.
         */
	public void setData(int maxNr) {
		String[] data;
		if (maxNr == 0)
			data = new String[] { "" };
		else {
			data = new String[maxNr];
			for (int i = 0; i < maxNr; i++)
				data[i] = "" + (i + 1);
		}
		iterlist.setData(data);
		oldnumber = this.getNumberOfLoops(false);
	}

	public void actionPerformed(ActionEvent e) {
		int old = oldnumber;
		int newNumber = -1;
		if (testInput(loopsfield.getText())) {
			newNumber = getNumberOfLoops(true);
			if (oldnumber != newNumber) {
				if (newNumber==1){
					newNumber=0;
				}
					
				setNumberOfLoops(newNumber);
				firePropertyChange("loop_number", old, newNumber);
			}
		}

	}

	public void addPropertyChangeListener(PropertyChangeListener pCL) {
		super.addPropertyChangeListener(pCL);
		// the view also has to react on spinenrevents
		iterlist.addPropertyChangeListener(pCL);
	}

	private boolean testInput(String value) {
		try {
			int i = Integer.parseInt(value);
			if (i < 0)
				throw new Exception();
		} catch (Exception e) {
			M4Interface.print.doPrint(Print.ERROR,Resource.getString("STEPSETTINGS_LOOPNUMBER_ERROR")+"\n"+e.getMessage(),e);
			JOptionPane.showMessageDialog(
				app,
				Resource.getString("STEPSETTINGS_LOOPNUMBER_ERROR"),
				Resource.getString("DIALOG_ERROR_TITLE"),
				JOptionPane.ERROR_MESSAGE);
			return false;
		}
		return true;
	}

}
/*
$Log: StepLoopPanel.java,v $
Revision 1.3  2006/04/11 14:10:13  euler
Updated license text.

Revision 1.2  2006/04/06 16:31:13  euler
Prepended license remark.

Revision 1.1  2006/01/03 09:54:15  hakenjos
Initial version!

*/
