/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.figures;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.ImageObserver;

import javax.swing.ImageIcon;

import org.jhotdraw.figures.LineFigure;
import org.jhotdraw.figures.RoundRectangleFigure;
import org.jhotdraw.framework.FigureChangeEvent;
import org.musoft.limo.application.Resource;
import org.musoft.limo.drawing.ModelDrawing;
import org.musoft.limo.drawing.ModelFigure;
import org.musoft.limo.model.ModelElement;
import org.musoft.limo.model.ModelFigureElement;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.gui.model.MiningMartConcept;
import edu.udo.cs.miningmart.m4.Concept;
import edu.udo.cs.miningmart.m4.M4Interface;
import edu.udo.cs.miningmart.m4.utils.Print;

/**
 * A <code>ModelFigure</code> for the <code>MiningMartConcept</code>.
 * @author Daniel Hakenjos
 * @version $Id: ConceptFigure.java,v 1.3 2006/04/11 14:10:17 euler Exp $
 */
public class ConceptFigure extends ModelFigure implements ImageObserver{

	private Concept concept;

	private Point origin;

	private LineFigure mainLine;
	
	private Image conceptimage;

	private final Color selected_color = Resource.getColor("CONCEPT_SELECTED_COLOR");
	private final Color not_selected_color = Resource.getColor("CONCEPT_NOTSELECTED_COLOR");
		
	private boolean isSelected;

	/**
	 * Constructs new <code>ConceptFigure</code>.
	 */
	public ConceptFigure(ModelFigureElement element, ModelDrawing drawing) {
		super(element, drawing);

		isSelected=false;
		//Concept holen
		concept=((MiningMartConcept) this.getElement()).getConcept();

		// Ellipse mit neuer Bounds zeichnen
		Rectangle bounds 	= this.getElement().getBounds();
		origin 		= bounds.getLocation();
		Point corner 		= new Point(origin.x + bounds.width, origin.y + bounds.height);	
		this.setPresentationFigure(new RoundRectangleFigure(origin, corner));
		
		// Label positionieren und initiallisieren		
		setLabelPosition(LABEL_INSIDE | LABEL_NORTH);
		setLabel(getElement().getName());
		
		// Hauptlinie zeichnen
		this.mainLine = new LineFigure(); 	
		this.add(mainLine);


		// Label positionieren und initiallisieren		
		setLabelPosition(LABEL_INSIDE | LABEL_NORTH);
		setLabel(getElement().getName());

		this.setSizeable(false);
		
		initConceptImage();
	}
	
	private void initConceptImage(){

		String img=null;
		if (concept.getType().equals(Concept.TYPE_MINING)){
			img=Resource.getString("CONCEPT_MINING_ICON");
		}else{
			
			try {
				if (concept.getColumnSets().size()>0){
					img=Resource.getString("CONCEPT_DB_CONNECTED_ICON");
				}else{
					img=Resource.getString("CONCEPT_DB_NOTCONNECTED_ICON");
				}
			} catch (M4Exception error) {
				M4Interface.print.doPrint(Print.ERROR, error.getMessage(),error);
				img=Resource.getString("CONCEPT_DB_NOTCONNECTED_ICON");
			}
		}
		
		conceptimage=(new ImageIcon(this.getClass().getResource(img))).getImage();
	}


	/**
	 * Setzt den Namen der Figure in das Label der Figure. Ob der Name angezeigt
	 * wird oder nicht, wird durch die Figure selber entschieden.
	 * 
	 * @param ModelElement sender 
	 */
	public void onSetName(ModelElement sender) {
		if (sender == getElement()) {
			setLabel(sender.getName());
		}
	}

	/**
	 * Modifiziert die Darstellung des Statchartfigurs so, dass es aktive
	 * erscheint. 
	 */
	public void isSelected() {
		isSelected=true;
		update();
	}
	
	/**
	 * Modifiziert die Darstellung des Statchartfigurs so, dass es inaktive
	 * erscheint. 
	 */
	public void deSelected() {
		isSelected=false;
		update();
	}
    


	/**
	 *Gets the <code>MiningMartConcept</code>.
	 */
	public MiningMartConcept getMiningMartConcept() {
		return (MiningMartConcept) getElement();
	}

	/* (non-Javadoc)
	 * @see org.jhotdraw.framework.Figure#moveBy(int, int)
	 */
	public void moveBy(int dx, int dy) {
		if ((dx==0)&&(dy==0)){
			return;
		}
		super.moveBy(dx,dy);
		origin=new Point(origin.x+dx,origin.y+dy);

		MiningMartConcept concept=((MiningMartConcept) getElement());
		try{
			concept.getConcept().setPoint(origin);
		}catch(M4Exception error){
			M4Interface.print.doPrint(Print.ERROR, error.getMessage(),error);
		}
	}

	
	public void draw(Graphics g) {
		super.draw(g);

		Rectangle rect = displayBox();
		
		if (isSelected){
			g.setColor(this.selected_color);
		}else{
			g.setColor(this.not_selected_color);
		}
		g.fillRoundRect(rect.x,rect.y,rect.width,rect.height,10,10);		

		this.getLabelFigure().draw(g);
		
		g.setColor(Color.BLACK);
		g.drawRoundRect(rect.x,rect.y,rect.width,rect.height,10,10);		
		g.drawLine(rect.x, rect.y + 20, rect.x + rect.width-1, rect.y + 20);
		
		g.drawLine(rect.x+30,rect.y+20,rect.x+30,rect.y+rect.height-1);

		this.initConceptImage();
		if (conceptimage != null) {
			g.drawImage(conceptimage, origin.x+3, origin.y+23, 24,24, this);
		}
		
		
		try {
			int nr_bas=concept.getAllBaseAttributes().size();
			int nr_mcfs=concept.getAllMultiColumnFeatures().size();
			int nr_colsets=concept.getColumnSets().size();
			
			Font font=g.getFont().deriveFont(Font.PLAIN);
			g.setFont(font);
			
			g.drawString(Integer.toString(nr_bas)+" BaseAttributes",rect.x+35,rect.y+35);
			g.drawString(Integer.toString(nr_mcfs)+" MultiColumnFeatures",rect.x+35,rect.y+50);
			g.drawString(Integer.toString(nr_colsets)+" ColumnSets",rect.x+35,rect.y+65);
		} catch (M4Exception error) {
			M4Interface.print.doPrint(Print.ERROR, error.getMessage(),error);
		}
	}

	/* (non-Javadoc)
	 * @see java.awt.image.ImageObserver#imageUpdate(java.awt.Image, int, int, int, int, int)
	 */
	public boolean imageUpdate(Image img, int flags, int x, int y, int width, int height) {
			if ((flags & (FRAMEBITS|ALLBITS)) != 0) {
				invalidate();
				if (listener() != null) {
					listener().figureRequestUpdate(new FigureChangeEvent(this));
				}
			}
			return (flags & (ALLBITS|ABORT)) == 0;
	}

}
/*
 * Historie
 * --------
 *
 * $Log: ConceptFigure.java,v $
 * Revision 1.3  2006/04/11 14:10:17  euler
 * Updated license text.
 *
 * Revision 1.2  2006/04/06 16:31:16  euler
 * Prepended license remark.
 *
 * Revision 1.1  2006/01/03 09:54:33  hakenjos
 * Initial version!
 *
 */
