/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.concepteditor;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.Collections;
import java.util.Vector;

import javax.swing.AbstractListModel;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.SpringLayout;
import javax.swing.SwingConstants;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumnModel;

import org.musoft.limo.application.Resource;

import edu.udo.cs.miningmart.db.DB;
import edu.udo.cs.miningmart.exception.DbConnectionClosed;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.gui.application.MiningMartApplication;
import edu.udo.cs.miningmart.gui.stepsettings.ListSelectDialog;
import edu.udo.cs.miningmart.gui.util.M4ObjectComparator;
import edu.udo.cs.miningmart.m4.Column;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.M4Interface;
import edu.udo.cs.miningmart.m4.RelationalDatatypes;
import edu.udo.cs.miningmart.m4.utils.Print;

/**
 * A simple <code>JDialog</code> to view the data of a <Columnset</code>.
 * 
 * @author Daniel Hakenjos
 * @version $Id: DataDialog.java,v 1.4 2006/04/11 14:10:17 euler Exp $
 */
public class DataDialog extends JDialog implements ActionListener, ItemListener{

	private MiningMartApplication app;

	private Columnset columnset;
	
	private Vector columns, selectedcolumns, sortcolumns;
	
	private JButton closeButton,updateButton, selectButton, nextButton, sortButton;
	
	private JTextField startrowField,endrowField;
	
	private JList columnlist, sortlist;
	
	private JTable datatable;

	private int nr_cols, nr_rows;

	//Object[nr_rows][nr_cols]	
	private Object[][] data;
	private ResultSetMetaData metadata;
	
	private int startRow;

	private JComboBox sortBox;

	public DataDialog(MiningMartApplication app, Columnset columnset){
		super(app,true);
		this.app=app;
		this.columnset=columnset;

		try {
			// Not necessary for just displaying the data:
			// this.columnset.updateStatistics();
			
			this.columns=new Vector();
			this.selectedcolumns=new Vector();
			this.sortcolumns=new Vector();
			this.columns.addAll(columnset.getColumns());
			this.selectedcolumns.addAll(columnset.getColumns());
			Collections.sort(columns,new M4ObjectComparator());
			Collections.sort(selectedcolumns,new M4ObjectComparator());
		} catch (M4Exception error) {
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
		}
		
		initComponents();
		this.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
	
		this.pack();
	
		setSize(800, 600);
	
		Dimension size = this.getSize();
	
		this.setLocation(
			(Toolkit.getDefaultToolkit().getScreenSize().width - size.width)
				/ 2,
			(Toolkit.getDefaultToolkit().getScreenSize().height - size.height)
				/ 2);
		this.setVisible(true);
	}
	
	private void initComponents(){

		JPanel toppanel=new JPanel();
		toppanel.setLayout(new BorderLayout(5,5));
		toppanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
		this.getContentPane().add(toppanel);

		//Info
		JPanel infopanel=new JPanel();
		infopanel.setLayout(new GridLayout(2,1,5,5));
		infopanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
		toppanel.add(infopanel,BorderLayout.NORTH);	

		JLabel label=new JLabel();
		label.setText(Resource.getString("COLUMNSET")+" - "+Resource.getString("DATA"));
		Font font=new Font("SansSerif",Font.BOLD,15);
		label.setFont(font);
		label.setHorizontalAlignment(SwingConstants.CENTER);	
		infopanel.add(label);

		label=new JLabel();
		label.setText(columnset.getName());
		font=new Font("SansSerif",Font.PLAIN,13);
		label.setFont(font);
		label.setHorizontalAlignment(SwingConstants.CENTER);	
		label.setForeground(Color.BLUE);
		infopanel.add(label);

		//Button		

		sortButton=new JButton(Resource.getString("DATA_SORT"));
		sortButton.addActionListener(this);
		
		nextButton=new JButton(Resource.getString("NEXT"));
		nextButton.addActionListener(this);
		
		updateButton=new JButton(Resource.getString("UPDATE"));
		updateButton.addActionListener(this);
		updateButton.setEnabled(false);			

		closeButton=new JButton(Resource.getString("CLOSE"));
		closeButton.addActionListener(this);

		JPanel centerpanel=new JPanel();
		centerpanel.setLayout(new BorderLayout(5,5));
		toppanel.add(centerpanel,BorderLayout.CENTER);
		
		//init the datatable
		startrowField=new JTextField(Resource.getString("DATA_FIRSTROW_NR"));
		endrowField=new JTextField(Resource.getString("DATA_LASTROW_NR"));
		initData();
		datatable=new JTable(){
			public TableCellRenderer getCellRenderer(int row, int column){
				return new DataRenderer();				
			}
		};
		datatable.setModel(new DataModel());
		datatable.setSelectionMode(javax.swing.ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		datatable.getTableHeader().setReorderingAllowed(false);
		datatable.setRowHeight(datatable.getRowHeight() + 2);
		TableColumnModel cmodel=datatable.getColumnModel();
		for(int i=0;i<cmodel.getColumnCount();i++){
			cmodel.getColumn(i).setMinWidth(10);
			cmodel.getColumn(i).setMaxWidth(Integer.MAX_VALUE);
		}
		datatable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		
		
		centerpanel.add(new JScrollPane(datatable),BorderLayout.CENTER);
		
		JPanel updatePanel=new JPanel();
		updatePanel.setLayout(new BorderLayout(5,5));
		centerpanel.add(updatePanel,BorderLayout.EAST);
		
		JPanel buttonPanel1=new JPanel();
		buttonPanel1.setLayout(new BoxLayout(buttonPanel1,BoxLayout.X_AXIS));
		buttonPanel1.add(Box.createHorizontalGlue());
		buttonPanel1.add(closeButton);
		updatePanel.add(buttonPanel1,BorderLayout.SOUTH);
		updatePanel.setPreferredSize(new Dimension(180,updatePanel.getHeight()));
		
		JPanel updatePanel2=new JPanel();
		updatePanel2.setLayout(new GridLayout(3,1,25,25));
		updatePanel.add(updatePanel2,BorderLayout.CENTER);

		//Column Selection
		JPanel columnPanel=new JPanel();
		columnPanel.setLayout(new BorderLayout(5,5));
		updatePanel2.add(columnPanel);
	
		selectButton=new JButton(Resource.getString("DATA_SELECT"));
		selectButton.addActionListener(this);

		JPanel buttonPanel3=new JPanel();
		buttonPanel3.setLayout(new BoxLayout(buttonPanel3,BoxLayout.X_AXIS));
		buttonPanel3.add(Box.createHorizontalGlue());
		buttonPanel3.add(selectButton);
		columnPanel.add(buttonPanel3,BorderLayout.SOUTH);

		columnlist=new JList(new ColumnListModel());
		columnlist.setVisibleRowCount(10);		
		columnPanel.add(new JScrollPane(columnlist), BorderLayout.CENTER);
		
		label=new JLabel(Resource.getString("COLUMNS")+":");
		columnPanel.add(label,BorderLayout.NORTH);

		//Sort values
		JPanel sortPanel=new JPanel();
		sortPanel.setLayout(new BorderLayout(5,5));
		updatePanel2.add(sortPanel);

		label=new JLabel(Resource.getString("DATA_SORT_BY"));
		sortPanel.add(label,BorderLayout.NORTH);
		JPanel buttonPanel4=new JPanel();
		
		sortlist=new JList(new SortListModel());
		sortlist.setVisibleRowCount(5);		
		sortPanel.add(new JScrollPane(sortlist), BorderLayout.CENTER);
		
		buttonPanel4.setLayout(new BoxLayout(buttonPanel4,BoxLayout.X_AXIS));
		buttonPanel4.add(Box.createHorizontalGlue());
		buttonPanel4.add(sortButton);

		Object[] sortvalues=new Object[]{Resource.getString("DATA_SORT_ASC"), Resource.getString("DATA-SORT_DESC")};
		sortBox=new JComboBox(sortvalues);
		sortBox.addItemListener(this);
		
		JPanel sortPanel2=new JPanel();
		sortPanel2.setLayout(new GridLayout(2,1,5,5));
		sortPanel2.add(sortBox);
		sortPanel2.add(buttonPanel4);
		sortPanel.add(sortPanel2,BorderLayout.SOUTH);
		


		//Row Selection
		JPanel rowPanel=new JPanel();
		rowPanel.setLayout(new SpringLayout());

		//startrowField und endrowField sind schon initialisiert		
		label=new JLabel(Resource.getString("DATA_FIRSTROW"));
		rowPanel.add(label);
		rowPanel.add(startrowField);
		label=new JLabel(Resource.getString("DATA_LASTROW"));
		rowPanel.add(label);
		rowPanel.add(endrowField);
		//		  Lay out the panel.
		SpringUtilities.makeCompactGrid(rowPanel,
										2, 2, //rows, cols
										6, 6,        //initX, initY
										6, 6);       //xPad, yPad
										
		//DocumentListener for TextFields
		startrowField.getDocument().addDocumentListener(new DocumentListener(){
			public void changedUpdate(DocumentEvent e) {
				updateButton.setEnabled(true);
			}
			public void insertUpdate(DocumentEvent e) {
				updateButton.setEnabled(true);
			}
			public void removeUpdate(DocumentEvent e) {
				updateButton.setEnabled(true);
			}
		});
		endrowField.getDocument().addDocumentListener(new DocumentListener(){
			public void changedUpdate(DocumentEvent e) {
				updateButton.setEnabled(true);
			}
			public void insertUpdate(DocumentEvent e) {
				updateButton.setEnabled(true);
			}
			public void removeUpdate(DocumentEvent e) {
				updateButton.setEnabled(true);
			}
		});
										
		JPanel rowPanel2=new JPanel();
		rowPanel2.setLayout(new BorderLayout(5,5));
		rowPanel2.add(rowPanel,BorderLayout.NORTH);
		updatePanel2.add(rowPanel2);				
		
		JPanel rowPanel3=new JPanel();
		rowPanel3.setLayout(new BorderLayout(5,5));
		rowPanel2.add(rowPanel3,BorderLayout.CENTER);
		
		JPanel buttonPanel2=new JPanel();
		buttonPanel2.setLayout(new BoxLayout(buttonPanel2,BoxLayout.X_AXIS));
		buttonPanel2.add(Box.createHorizontalGlue());
		buttonPanel2.add(nextButton);
		buttonPanel2.add(Box.createRigidArea(new Dimension(5, 0)));		
		buttonPanel2.add(updateButton);
		rowPanel3.add(buttonPanel2,BorderLayout.NORTH);

	}
	
	private void sortColumns(){
		String[] data=new String[selectedcolumns.size()];		
		for(int i=0;i<selectedcolumns.size();i++){
			data[i]=((Column) selectedcolumns.get(i)).getName();			
		}

		String[] selected_data=new String[sortcolumns.size()];
		for(int i=0;i<sortcolumns.size();i++){
			selected_data[i]=((Column) sortcolumns.get(i)).getName();			
		}

		ListSelectDialog dialog=new ListSelectDialog(app,Resource.getString("DATA_SORTDIALOG_TITLE"),Resource.getString("DATA_SORTDIALOG_SUBTITLE"),data,selected_data,true,0,Integer.MAX_VALUE);
		if (dialog.getExitAction()==ListSelectDialog.OK){
			selected_data=dialog.getSelectedDataItems();

			sortcolumns=new Vector();
			Column column;
			for(int i=0;i<selected_data.length;i++){
				try {
					column=columnset.getColumn(selected_data[i]);
					sortcolumns.add(column);				
				} catch (M4Exception error) {
					M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
				}
			}
			
			this.sortlist.setModel(new SortListModel());
			updateButton.setEnabled(true);		
			updateDataView();
		}		
	}
	
	private void nextDataView(){
		int startrow=0,endrow=0;
		
		try{
			startrow=Integer.parseInt(startrowField.getText());
			endrow=Integer.parseInt(endrowField.getText());
		}catch(NumberFormatException error){
			startrow=0;
			endrow=0;
		}
		
		startrow=Math.max(1,startrow);
		endrow=Math.max(1,endrow);
		startrow=Math.min(startrow,endrow);

		nr_rows=endrow-startrow+1;
		startrow+=nr_rows;
		endrow+=nr_rows;

		try {
			startrow=Math.min(startrow,columnset.getStatisticsAll());
			endrow=Math.min(endrow,columnset.getStatisticsAll());		
		} catch (M4Exception error) {
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
			return;
		}		

		this.startRow=startrow;
		startrowField.setText(Integer.toString(startrow));
		endrowField.setText(Integer.toString(endrow));
		updateDataView();
	}
	
	private boolean initData(){
		int startrow=0,endrow=0;
		
		try{
			startrow=Integer.parseInt(startrowField.getText());
			endrow=Integer.parseInt(endrowField.getText());
		}catch(NumberFormatException error){
			startrow=0;
			endrow=0;
		}
		
		startrow=Math.max(1,startrow);
		endrow=Math.max(1,endrow);
		startrow=Math.min(startrow,endrow);

		try {
			startrow=Math.min(startrow,columnset.getStatisticsAll());
			endrow=Math.min(endrow,columnset.getStatisticsAll());		
		} catch (M4Exception error) {
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
			return false;
		}		

		this.startRow=startrow;
		startrowField.setText(Integer.toString(startrow));
		endrowField.setText(Integer.toString(endrow));
		
		nr_cols=selectedcolumns.size();
		nr_rows=endrow-startrow+1;
		
		if ((nr_rows==0)||(nr_cols==0))
			return false;
			

		int dbms=0;
		try {
			dbms = M4Interface.getInstance().getM4db().getBusinessDbms();
		} catch (DbConnectionClosed error) {
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
			return false;
		}
		

		StringBuffer sbuf = new StringBuffer("SELECT ");
		Column column;
		for(int i=0;i<nr_cols;i++){
			if (i!=0){
				sbuf.append(", ");
			}
			column=(Column) selectedcolumns.get(i);				
			String sqlDef = column.getSQLDefinition();
			String colName = column.getName();
			sbuf.append(sqlDef);
			if (! sqlDef.equals(colName)) {
				sbuf.append(" AS " + colName);
			}
		}

		/*		if ((dbms==DB.POSTGRES)&&(columnset.getType().equals(Columnset.CS_TYPE_VIEW))){
				}else{
				}*/
		String sql=null;
		sql = sbuf.toString()
				   + " FROM " + columnset.getSchemaPlusName();
		if (sortcolumns.size()>0){
			sql += " ORDER BY ";

			for(int i=0;i<sortcolumns.size();i++){
				if (i!=0){
					sql+=", ";
				}
				column=(Column) sortcolumns.get(i);				
				sql+=column.getSQLDefinition();
			}
			if (sortBox.getSelectedIndex()==0){
				sql+=" ASC";
			}else{
				sql+=" DESC";
			}
		}
		
		if (sql==null)
			return false;

		//M4Interface.print.doPrint(Print.OPERATOR,sql);

		ResultSet resultset=null;
		try {
			resultset=M4Interface.getCurrentCase().executeBusinessSqlRead(sql);
		} catch (M4Exception error) {
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
			return false;
		}

		try {
			this.metadata=resultset.getMetaData();
		} catch (SQLException error) {
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
			return false;
		}

		try {
			this.nr_cols=metadata.getColumnCount();
		} catch (SQLException error) {
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
			return false;
		}
		nr_rows=endrow-startrow+1;
		
		this.data=new Object[nr_rows][nr_cols];

		//fill the data
		try {
			//resultset.beforeFirst();

			//read the data for PostgreSQL if Columnset is a view
			if ((dbms==DB.POSTGRES)&&(columnset.getType().equals(Columnset.CS_TYPE_VIEW))){
				for(int i=1;i<startrow;i++){
					resultset.next();
				}
				int row=0;
				while ((resultset.next())&&(row<nr_rows)){
					for(int i=0;i<nr_cols;i++){
						data[row][i]=resultset.getObject(i+1);
					}
					row++;
				}
			}else{
				for(int i=1;i<startrow;i++){
					resultset.next();
				}
				int row=0;
				while((resultset.next())&&(row<nr_rows)){
					for(int i=0;i<nr_cols;i++){
						data[row][i]=resultset.getObject(i+1);
					}
					row++;
				}
				//System.out.println("Rows in ResultSet: "+(row+1));
			}
		
		} catch (SQLException error) {
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
			return false;
		}
		
		
		try {
			resultset.close();
		} catch (SQLException error) {
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
			return false;
		}
		return true;
	}
	
	private void updateDataView(){
		
		if (initData()){
			datatable.setModel(new DataModel());
			updateButton.setEnabled(false);		
		}
	}
	
	private void selectColumns(){
		String[] data=new String[columns.size()];		
		for(int i=0;i<columns.size();i++){
			data[i]=((Column) columns.get(i)).getName();			
		}

		String[] selected_data=new String[selectedcolumns.size()];
		for(int i=0;i<selectedcolumns.size();i++){
			selected_data[i]=((Column) selectedcolumns.get(i)).getName();			
		}

		ListSelectDialog dialog=new ListSelectDialog(app,Resource.getString("DATA_SELECTDIALOG_TITLE"),Resource.getString("DATA_SELECTDIALOG_SUBTITLE"),data,selected_data,true,0,Integer.MAX_VALUE);
		if (dialog.getExitAction()==ListSelectDialog.OK){
			selected_data=dialog.getSelectedDataItems();

			selectedcolumns=new Vector();
			Column column;
			for(int i=0;i<selected_data.length;i++){
				try {
					column=columnset.getColumn(selected_data[i]);
					selectedcolumns.add(column);				
				} catch (M4Exception error) {
					M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
				}
			}
			
			//the sort columns have to be selected
			boolean removed=false;
			for(int i=sortcolumns.size()-1;i>=0;i--){
				column=(Column) sortcolumns.get(i);
				if (!selectedcolumns.contains(column)){
					sortcolumns.remove(i);								
					removed=true;
				}
			}
			if (removed){
				this.sortlist.setModel(new SortListModel());
			}
			
			this.columnlist.setModel(new ColumnListModel());
			updateButton.setEnabled(true);		
			updateDataView();
		}		
	}

	/* (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent event) {
		if (!(event.getSource() instanceof JButton))
			return;
		JButton source=(JButton) event.getSource();
		
		if (source.getText().equals(closeButton.getText())){
			dispose();
		}
		if (source.getText().equals(nextButton.getText())){
			nextDataView();
		}
		if (source.getText().equals(updateButton.getText())){
			updateDataView();
		}
		if (source.getText().equals(selectButton.getText())){
			selectColumns();
		}
		if (source.getText().equals(sortButton.getText())){
			sortColumns();
		}
	}

	/* (non-Javadoc)
	 * @see java.awt.event.ItemListener#itemStateChanged(java.awt.event.ItemEvent)
	 */
	public void itemStateChanged(ItemEvent e) {
		updateDataView();
	}

	
	private class SortListModel extends AbstractListModel{
		
		/* (non-Javadoc)
		 * @see javax.swing.ListModel#getElementAt(int)
		 */
		public Object getElementAt(int index) {
			return (String) ((Column) sortcolumns.get(index)).getName();
		}

		/* (non-Javadoc)
		 * @see javax.swing.ListModel#getSize()
		 */
		public int getSize() {
			return sortcolumns.size();
		}

	}
	
	private class ColumnListModel extends AbstractListModel{
		
		/* (non-Javadoc)
		 * @see javax.swing.ListModel#getElementAt(int)
		 */
		public Object getElementAt(int index) {
			return (String) ((Column) selectedcolumns.get(index)).getName();
		}

		/* (non-Javadoc)
		 * @see javax.swing.ListModel#getSize()
		 */
		public int getSize() {
			return selectedcolumns.size();
		}

	}
	
	private class DataRenderer extends DefaultTableCellRenderer{
		
			/* (non-Javadoc)
		 * @see javax.swing.table.TableCellRenderer#getTableCellRendererComponent(javax.swing.JTable, java.lang.Object, boolean, boolean, int, int)
		 */
		public Component getTableCellRendererComponent(
			JTable table,
			Object value,
			boolean isSelected,
			boolean hasFocus,
			int row,
			int column) {
				
			Component comp=super.getTableCellRendererComponent(
				table,value, isSelected, hasFocus, row, column);

			if ((row+1)%2==0){
				comp.setBackground(Resource.getColor("CONCEPT_NOTSELECTED_COLOR"));				
			}
			
			return comp;
		}

	}
	
	private class DataModel extends AbstractTableModel{
		
		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnClass(int)
		 */
		public Class getColumnClass(int columnIndex) {
			if (columnIndex==0){
				return Integer.class;
			}
			
			Column column=null;
			try {
				column=columnset.getColumn(getColumnName(columnIndex-1));
			} catch (M4Exception error) {
				M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
			}

			String datatype=null;
			try {
				datatype = column.getColumnDataTypeName();
			} catch (DbConnectionClosed error) {
				M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
				datatype=null;
			} catch (M4Exception error) {
				M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
				datatype=null;
			}
			if (datatype==null){
				return String.class;
			}
			
			if (datatype.equals(RelationalDatatypes.RELATIONAL_DATATYPE_DATE)){
				return String.class;
			}else if (datatype.equals(RelationalDatatypes.RELATIONAL_DATATYPE_KEY)){
				return String.class;
			}else if (datatype.equals(RelationalDatatypes.RELATIONAL_DATATYPE_NUMBER)){
				return Double.class;
			}else if (datatype.equals(RelationalDatatypes.RELATIONAL_DATATYPE_STRING)){
				return String.class;	
			}else{
				return String.class; 
			}
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnName(int)
		 */
		public String getColumnName(int column) {
			if (column==0){
				return Resource.getString("DATA_ROW");
			}
			try {
				return metadata.getColumnName(column);
			} catch (SQLException e) {
			}
			return "";
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#isCellEditable(int, int)
		 */
		public boolean isCellEditable(int rowIndex, int columnIndex) {
			return false;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnCount()
		 */
		public int getColumnCount() {
			return nr_cols + 1;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getRowCount()
		 */
		public int getRowCount() {
			return nr_rows;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getValueAt(int, int)
		 */
		public Object getValueAt(int rowIndex, int columnIndex) {
			if (columnIndex==0){
				return new Integer(startRow+rowIndex);
			}
			return data[rowIndex][columnIndex-1];
		}
		
	}



}
/*
 * Historie
 * --------
 *
 * $Log: DataDialog.java,v $
 * Revision 1.4  2006/04/11 14:10:17  euler
 * Updated license text.
 *
 * Revision 1.3  2006/04/06 16:31:16  euler
 * Prepended license remark.
 *
 * Revision 1.2  2006/01/13 01:19:46  scholz
 * improved statistics computation
 *
 * Revision 1.1  2006/01/03 09:54:25  hakenjos
 * Initial version!
 *
 */

/*
Revision 1.5  2005/12/22 10:46:21  hakenjos
*** empty log message ***

Revision 1.4  2005/11/24 09:50:28  hakenjos
Names etc. replaced with resource-access

Revision 1.3  2005/11/17 11:01:11  hakenjos
*** empty log message ***

Revision 1.2  2005/11/17 10:57:26  hakenjos
*** empty log message ***

 */
