/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.application;

import java.io.IOException;
import java.io.OutputStream;
import java.util.logging.ErrorManager;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.LogManager;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

import edu.udo.cs.miningmart.m4.utils.PrintFormatter;

/**
 * A <code>Handler</code> for the <code>PrintViewer</code>.
 * @author Daniel Hakenjos
 * @version $Id: PrintViewHandler.java,v 1.3 2006/04/11 14:10:15 euler Exp $
 */
public class PrintViewHandler extends Handler {

	private LogManager manager = LogManager.getLogManager();
	private OutputStream output;
	private boolean doneHeader;

	// Private method to configure a StreamHandler from LogManager
	// properties and/or default values as specified in the class
	// javadoc.
	private void configure(Formatter formatter) {
		LogManager manager = LogManager.getLogManager();
		String cname = getClass().getName();
		Logger logger=Logger.getLogger("org.musoft.limo");

		setLevel(logger.getLevel());
		setFilter(logger.getFilter());
		setFormatter(new PrintFormatter());
		try {
			setEncoding(manager.getProperty(cname + ".encoding"));
		} catch (Exception ex) {
			try {
				setEncoding(null);
			} catch (Exception ex2) {
				// doing a setEncoding with null should always work.
				// assert false;
			}
		}
	}

	/**
	 * Create a <tt>StreamHandler</tt> with a given <tt>Formatter</tt>
	 * and output stream.
	 * <p>
	 * @param out         the target output stream
	 * @param formatter   Formatter to be used to format output
	 */
	public PrintViewHandler(OutputStream out) {
		configure(new PrintFormatter());
		this.output=out;
	}

	/**
	 * Create a <tt>StreamHandler</tt> with a given <tt>Formatter</tt>
	 * and output stream.
	 * <p>
	 * @param out         the target output stream
	 * @param formatter   Formatter to be used to format output
	 */
	public PrintViewHandler(OutputStream out, Formatter formatter) {
		configure(formatter);
		this.output=out;
	}


	/**
	 * Format and publish a <tt>LogRecord</tt>.
	 * <p>
	 * The <tt>StreamHandler</tt> first checks if there is an <tt>OutputStream</tt>
	 * and if the given <tt>LogRecord</tt> has at least the required log level.
	 * If not it silently returns.  If so, it calls any associated
	 * <tt>Filter</tt> to check if the record should be published.  If so,
	 * it calls its <tt>Formatter</tt> to format the record and then writes 
	 * the result to the current output stream.
	 * <p>
	 * If this is the first <tt>LogRecord</tt> to be written to a given
	 * <tt>OutputStream</tt>, the <tt>Formatter</tt>'s "head" string is 
	 * written to the stream before the <tt>LogRecord</tt> is written.
	 *
	 * @param  record  description of the log event. A null record is
	 *                 silently ignored and is not published
	 */
	public synchronized void publish(LogRecord record) {
		if (!isLoggable(record)) {
			return;
		}
		String msg;
		try {
			msg = getFormatter().format(record);
		} catch (Exception ex) {
			// We don't want to throw an exception here, but we
			// report the exception to any registered ErrorManager.
			reportError(null, ex, ErrorManager.FORMAT_FAILURE);
			return;
		}

		try {
			if (!doneHeader) {
				output.write(getFormatter().getHead(this).getBytes());
				doneHeader = true;
			}
			output.write(msg.getBytes());
		} catch (Exception ex) {
			// We don't want to throw an exception here, but we
			// report the exception to any registered ErrorManager.
			reportError(null, ex, ErrorManager.WRITE_FAILURE);
		}
	}
	
	

	/* (non-Javadoc)
	 * @see java.util.logging.Handler#close()
	 */
	public void close() throws SecurityException {
		try{
			output.close();
		}catch(IOException error){
		}
	}

	/* (non-Javadoc)
	 * @see java.util.logging.Handler#flush()
	 */
	public void flush() {
		try{
			output.flush();
		}catch(IOException error){
		}
	}

}
/*
 * Historie
 * --------
 *
 * $Log: PrintViewHandler.java,v $
 * Revision 1.3  2006/04/11 14:10:15  euler
 * Updated license text.
 *
 * Revision 1.2  2006/04/06 16:31:14  euler
 * Prepended license remark.
 *
 * Revision 1.1  2006/01/03 09:54:27  hakenjos
 * Initial version!
 *
 */

/*
Revision 1.2  2005/06/07 09:27:57  hakenjos
Changes made for printing erros, messaged etc.
All properties are now in the properties-files.

Revision 1.1  2005/05/24 07:02:36  hakenjos
A new Handler for the PrintViewer

*/
