/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.application;

import org.jhotdraw.framework.DrawingView;
import org.jhotdraw.framework.Figure;
import org.jhotdraw.framework.FigureEnumeration;
import org.jhotdraw.framework.FigureSelectionListener;

import edu.udo.cs.miningmart.gui.figures.ChainFigure;
import edu.udo.cs.miningmart.gui.figures.ConceptFigure;
import edu.udo.cs.miningmart.gui.figures.FeatureFigure;
import edu.udo.cs.miningmart.gui.figures.ProjectionFigure;
import edu.udo.cs.miningmart.gui.figures.RelationFigure;
import edu.udo.cs.miningmart.gui.figures.StepFigure;
import edu.udo.cs.miningmart.gui.figures.SubConceptFigure;
import edu.udo.cs.miningmart.gui.figures.TransitionFigure;
import edu.udo.cs.miningmart.gui.model.MiningMartBaseAttribute;
import edu.udo.cs.miningmart.gui.model.MiningMartChain;
import edu.udo.cs.miningmart.gui.model.MiningMartConcept;
import edu.udo.cs.miningmart.gui.model.MiningMartMultiColumnFeature;
import edu.udo.cs.miningmart.gui.model.MiningMartProjection;
import edu.udo.cs.miningmart.gui.model.MiningMartRelation;
import edu.udo.cs.miningmart.gui.model.MiningMartStep;
import edu.udo.cs.miningmart.gui.model.MiningMartSubConcept;
import edu.udo.cs.miningmart.gui.model.StepTransition;

/**
 * Listener for the selection in a MiningMartDrawingView.
 * So you can get the ModelElemets for the selected firgures.
 * @author Daniel Hakenjos
 * @version $Id: MiningMartSelectionListener.java,v 1.4 2006/04/11 14:10:15 euler Exp $
 */
public class MiningMartSelectionListener implements FigureSelectionListener {

	private MiningMartApplication app;

	private MiningMartStep mstep;
	private MiningMartChain mchain;
	private StepTransition steptrans;
	
	private MiningMartConcept mconcept;
	private MiningMartBaseAttribute mbaseatt;
	private MiningMartMultiColumnFeature mmcfeat;
	
	private MiningMartSubConcept msubconcept;
	private MiningMartRelation mrelation;
	private MiningMartProjection mprojection;
	
	private ConceptFigure conceptfigure;

	private boolean stepselected, chainselected, transselected;
	
	private boolean conceptselected, baseattributeselected, mcfeatselected;
	
	private boolean subconceptselected, projectionselected, relationselected;


	public MiningMartSelectionListener(MiningMartApplication app) {
		this.app = app;
		initSelection();
	}
	
	private void initSelection(){
		this.mstep = null;
		this.mchain=null;
		this.steptrans=null;
		this.mconcept=null;
		this.mbaseatt=null;
		this.mmcfeat=null;
		this.msubconcept=null;
		this.mprojection=null;
		this.mrelation=null;
		this.stepselected=false;
		this.chainselected=false;
		this.transselected=false;
		this.conceptselected=false;
		this.baseattributeselected=false;
		this.mcfeatselected=false;
		this.subconceptselected=false;
		this.projectionselected=false;
		this.relationselected=false;
	}

	/**
	 * @see CH.ifa.draw.framework.FigureSelectionListener#figureSelectionChanged(DrawingView)
	 */
	public void figureSelectionChanged(DrawingView view) {
		//init the selection
		initSelection();

		FigureEnumeration fenum = view.selection();
		
		if (fenum.hasNextFigure()) {
			Figure fig = fenum.nextFigure();
			if (fig instanceof ChainFigure) {
				chainselected=true;
				mchain=((ChainFigure) fig).getMiningMartChain();
			}else if (fig instanceof StepFigure){
				stepselected=true;
				mstep=((StepFigure) fig).getMiningMartStep();
			} else if (fig instanceof TransitionFigure) {
				transselected=true;
				steptrans=((TransitionFigure) fig).getStepTransition();
			} else if (fig instanceof ConceptFigure) {
				conceptselected=true;
				if (conceptfigure!=null){
					conceptfigure.deSelected();
				}
				conceptfigure=(ConceptFigure) fig;
				conceptfigure.isSelected();
				mconcept=conceptfigure.getMiningMartConcept();
			} else if ((fig instanceof FeatureFigure)&&(((FeatureFigure) fig).getElement() instanceof MiningMartBaseAttribute)) {
				baseattributeselected=true;
				mbaseatt=((FeatureFigure) fig).getMiningMartBaseAttribute();
			} else if ((fig instanceof FeatureFigure)&&(((FeatureFigure) fig).getElement() instanceof MiningMartMultiColumnFeature)) {
				mcfeatselected=true;
				mmcfeat=((FeatureFigure) fig).getMiningMartMultiColumnFeature();
			}else if (fig instanceof SubConceptFigure){
				subconceptselected=true;	
				msubconcept=((SubConceptFigure) fig).getMiningMartSubConcept();			
			}else if (fig instanceof RelationFigure){
				relationselected=true;
				mrelation=((RelationFigure) fig).getMiningMartRelation();
			}else if (fig instanceof ProjectionFigure){
				projectionselected=true;
				mprojection=((ProjectionFigure) fig).getMiningMartProjection();
			}
		}
	}

	/**
	 * Is a transition of a SubConcept selected?
	 */
	public boolean isMiningMartSubConceptSelected(){
		return subconceptselected;
	}

	/**
	 * Is a transition of a MiningMartRelation selected?
	 */
	public boolean isMiningMartRelationSelected(){
		return relationselected;
	}

	/**
	 * Is a transition of a MiningMartProjection selected?
	 */
	public boolean isMiningMartProjectionSelected(){
		return projectionselected;
	}

	/**
	 * Is a figure of a MiningMartConcept selected?
	 */
	public boolean isMiningMartConceptSelected(){
		return conceptselected;
	}

	/**
	 * Is a figure of a MiningMartBaseAttribute selected?
	 */
	public boolean isMiningMartBaseAttributeSelected(){
		return baseattributeselected;
	}
	
	/**
	 * Is a figure of a MiningMartMultiColumnFeature selected?
	 */
	public boolean isMiningMartMultiColumnFeatureSelected(){
		return mcfeatselected;
	}

	/**
	 * Is a figure of a MiningMartStep selected?
	 */
	public boolean isMiningMartStepSelected(){
		return stepselected;
	}
	
	/**
	 * Is a figure of a MiningMartChain selected?
	 */
	public boolean isMiningMartChainSelected(){
		return chainselected;
	}
	
	/**
	 * Is a figure of a StepTransition selected?
	 */
	public boolean isStepTransitionSelected(){
		return transselected;
	}
	
	public boolean isAnythingSelected(){
		if (app.getEditorMode()==MiningMartApplication.CASE_MODE){
			return (stepselected||chainselected||transselected);
		}else{
			return (conceptselected||baseattributeselected||mcfeatselected||subconceptselected||relationselected||projectionselected);
		}
	}

	/**
	 * Gets the selected MiningMartMultiColumnFeature.
	 * @return null, if no MiningMartMultiColumnFeature is selected
	 */
	public MiningMartMultiColumnFeature getSelectedMiningMartMultiColumnFeature() {
		if (!mcfeatselected)
			return null;
		return mmcfeat;
	}

	/**
	 * Gets the selected MiningMartBaseAttribute.
	 * @return null, if no MiningMartBaseAttribute is selected
	 */
	public MiningMartBaseAttribute getSelectedMiningMartBaseAttribute() {
		if (!baseattributeselected)
			return null;
		return mbaseatt;
	}

	/**
	 * Gets the selected MiningMartConcept.
	 * @return null, if no MiningMartConcept is selected
	 */
	public MiningMartConcept getSelectedMiningMartConcept() {
		if (!conceptselected)
			return null;
		return mconcept;
	}

	/**
	 * Gets the selected MiningMartStep.
	 * @return null, if no MiningMartStep is selected
	 */
	public MiningMartStep getSelectedMiningMartStep() {
		if (!stepselected)
			return null;
		return mstep;
	}

	/**
	 * Gets the selected MiningMartChain.
	 * @return null, if no MiningMartChain is selected
	 */
	public MiningMartChain getSelectedMiningMartChain() {
		if (!chainselected)
			return null;
		return mchain;
	}


	/**
	 * Gets the selected StepTransition.
	 * @return null, if no StepTransition is selected
	 */
	public StepTransition getSelectedStepTransition() {
		if (!transselected)
			return null;
		return steptrans;
	}
	
	/**
	 * Gets the selected MiningMartSubConcept.
	 * @return null, if no MiningMartSubConcept is selected
	 */
	public MiningMartSubConcept getSelectedMiningMartSubConcept(){
		if (!subconceptselected)
			return null;
		return msubconcept;
	}
	
	/**
	 * Gets the selected MiningMartRelation.
	 * @return null, if no MiningMartRelation is selected
	 */
	public MiningMartRelation getSelectedMiningMartRelation(){
		if (!relationselected)
			return null;
		return mrelation;
	}

	/**
	 * Gets the selected MiningMartProjection.
	 * @return null, if no MiningMartProjection is selected
	 */
	public MiningMartProjection getSelectedMiningMartProjection(){
		if (!projectionselected)
			return null;
		return mprojection;
	}

}
/*
$Log: MiningMartSelectionListener.java,v $
Revision 1.4  2006/04/11 14:10:15  euler
Updated license text.

Revision 1.3  2006/04/06 16:31:15  euler
Prepended license remark.

Revision 1.2  2006/01/03 11:29:14  hakenjos
Delete Concept, SubConcept, Relation and Projection in Contextmenu.

Revision 1.1  2006/01/03 09:54:26  hakenjos
Initial version!

*/
