/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.application;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collection;
import java.util.Iterator;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.border.Border;

import org.musoft.limo.application.Resource;
import org.musoft.limo.model.ModelFigureElement;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.gui.model.MiningMartChain;
import edu.udo.cs.miningmart.gui.model.MiningMartStep;
import edu.udo.cs.miningmart.m4.Chain;
import edu.udo.cs.miningmart.m4.M4Interface;
import edu.udo.cs.miningmart.m4.Step;
import edu.udo.cs.miningmart.m4.utils.Print;

/**
 * A dialog for showing all dependencies. This is necessary if the transition starts and/or goes to a chain.
 * @author Daniel Hakenjos
 * @version $Id: ExistingTransitionsDialog.java,v 1.3 2006/04/11 14:10:15 euler Exp $
 */
public class ExistingTransitionsDialog extends JDialog implements ActionListener{

	public static final int OK = 0;

	private JButton ok;

	private JList fromlist, tolist;

	private Chain fromchain, tochain;
	private Step fromstep, tostep;

	private Step[] fromsteps, tosteps;

	private String[] fromstepnames, tostepnames;

	private boolean chaintochain = false,chaintostep = false,steptochain = false, steptostep=false;

	private int exitAction = OK;
	
	private ModelFigureElement start, end;


	/**
	 * Constructor for ExistingTransitionsDialog.
	 */
	public ExistingTransitionsDialog(MiningMartApplication parent, ModelFigureElement start, ModelFigureElement end) {
		super(parent, Resource.getString("SHOW_ALL_TRANSITIONS"), true);
		this.start=start;
		this.end=end;

		if (start instanceof MiningMartChain){
			fromchain=((MiningMartChain) start).getChain();
			if (end instanceof MiningMartChain){
				tochain=((MiningMartChain) end).getChain();
				chaintochain=true;
			}
			if (end instanceof MiningMartStep){
				tostep=((MiningMartStep) end).getStep();
				chaintostep=true;
			}
		}
		if (start instanceof MiningMartStep){
			fromstep=((MiningMartStep) start).getStep();
			if (end instanceof MiningMartChain){
				tochain=((MiningMartChain) end).getChain();
				steptochain=true;
			}
			if (end instanceof MiningMartStep){
				steptostep=true;
			}
		}
		
		if (!steptostep){
			initComponents();		
			this.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
	
			this.pack();
	
			setSize(500, 350);
	
			Dimension size = this.getSize();
	
			this.setLocation(
				(Toolkit.getDefaultToolkit().getScreenSize().width - size.width)
					/ 2,
				(Toolkit.getDefaultToolkit().getScreenSize().height - size.height)
					/ 2);
			this.setVisible(true);
		}
	}

	private void initComponents() {
		try {
			
			Border emptyborder=BorderFactory.createEmptyBorder(10,10,10,10);
			Border emptyborder2=BorderFactory.createEmptyBorder(2,10,2,10);
			
			JPanel top_panel = new JPanel();
			top_panel.setLayout(new BorderLayout(5, 5));
			
			JLabel toplabel=new JLabel(Resource.getString("TRANSITION")+": "+start.getName()+" -> "+end.getName());
			top_panel.add(toplabel,BorderLayout.NORTH);

			//Init the lists

			fromlist = new JList();
			fromlist.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
			fromlist.setLayoutOrientation(JList.VERTICAL);
			fromlist.setVisibleRowCount(-1);

			tolist = new JList();
			tolist.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
			tolist.setLayoutOrientation(JList.VERTICAL);
			tolist.setVisibleRowCount(-1);

			if (chaintochain) {

				Collection col = fromchain.getAllTransitionsToChain(tochain);
				Iterator iter = col.iterator();

				this.fromsteps = new Step[col.size()];
				this.tosteps = new Step[col.size()];
				this.fromstepnames = new String[col.size()];
				int index = 0;
				Step[] steps;
				while (iter.hasNext()) {
					steps = (Step[]) iter.next();
					fromsteps[index] = steps[0];
					tosteps[index] = steps[1];

					fromstepnames[index] = steps[0].getName()+" -> "+steps[1].getName();
					index++;
				}

				tostepnames=new String[]{Resource.getString("TRANSITION_NONE")};
			}

			if (steptochain) {
				Collection col = tochain.getSuccessors(fromstep);
				Iterator iter = col.iterator();

				this.tosteps = new Step[col.size()];
				this.tostepnames = new String[col.size()];
				int index = 0;
				Step step;
				while (iter.hasNext()) {
					step = (Step) iter.next();
					tosteps[index] = step;
					tostepnames[index] = step.getName();

					index++;
				}

				fromsteps = new Step[] { tostep };
				fromstepnames=new String[]{fromstep.getName()};
			}
			if (chaintostep) {
				Collection col = fromchain.getPredecessors(tostep);
				Iterator iter = col.iterator();

				this.fromsteps = new Step[col.size()];
				this.fromstepnames = new String[col.size()];
				int index = 0;
				Step step;
				while (iter.hasNext()) {
					step = (Step) iter.next();
					fromsteps[index] = step;
					fromstepnames[index] = step.getName();

					index++;
				}

				tosteps = new Step[] { tostep };
				tostepnames=new String[]{tostep.getName()};
			}

			fromlist.setListData(fromstepnames);

			tolist.setListData(tostepnames);
			
			JScrollPane fromscrollpane = new JScrollPane(fromlist);
			JScrollPane toscrollpane = new JScrollPane(tolist);
			
			fromscrollpane.setBorder(emptyborder2);
			toscrollpane.setBorder(emptyborder2);

			JLabel fromlabel=new JLabel(Resource.getString("FROM_STEP"));
			JLabel tolabel=new JLabel(Resource.getString("TO_STEP"));
			fromlabel.setBorder(emptyborder);
			tolabel.setBorder(emptyborder);
			
			JPanel frompanel=new JPanel();
			frompanel.setLayout(new BorderLayout());
			frompanel.add(fromlabel,BorderLayout.NORTH);
			frompanel.add(fromscrollpane,BorderLayout.CENTER);

			JPanel topanel=new JPanel();
			topanel.setLayout(new BorderLayout());
			topanel.add(tolabel,BorderLayout.NORTH);
			topanel.add(toscrollpane,BorderLayout.CENTER);


			JPanel mid_panel = new JPanel();
			if (chaintochain){
				mid_panel.setLayout(new GridLayout(1,1));
			}
			else{
				mid_panel.setLayout(new GridLayout(1, 2));
			}

			mid_panel.add(frompanel);
			if (!chaintochain){
				mid_panel.add(topanel);
			}

			top_panel.add(mid_panel, BorderLayout.CENTER);

			ok = new JButton(Resource.getString("OK"));
			if ((fromstepnames.length==1)&&(tostepnames.length==1)){
				ok.setEnabled(true);
			}

			

			ok.addActionListener(this);

			JPanel bottom = new JPanel();
			bottom.setLayout(new BoxLayout(bottom, BoxLayout.X_AXIS));
			bottom.add(Box.createHorizontalGlue());
			bottom.add(ok);
			bottom.add(Box.createRigidArea(new Dimension(10, 0)));

			top_panel.add(bottom, BorderLayout.SOUTH);

			top_panel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
			this.getContentPane().add(top_panel);
		} catch (M4Exception error) {
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
			dispose();
		}
	}

	/**
	 * Gets the exit action.
	 * @see #OK
	 */
	public int getExitAction() {
		return exitAction;
	}

	/**
	 * @see java.awt.event.ActionListener#actionPerformed(ActionEvent)
	 */
	public void actionPerformed(ActionEvent e) {
		String cmd = e.getActionCommand();

		if (cmd.equals(Resource.getString("OK"))) {
			exitAction = OK;
			this.dispose();
		}
	}


}
/*
 * Historie
 * --------
 *
 * $Log: ExistingTransitionsDialog.java,v $
 * Revision 1.3  2006/04/11 14:10:15  euler
 * Updated license text.
 *
 * Revision 1.2  2006/04/06 16:31:15  euler
 * Prepended license remark.
 *
 * Revision 1.1  2006/01/03 09:54:27  hakenjos
 * Initial version!
 *
 */
