/*
 * MiningMart Version 1.0
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.compiler;

import java.rmi.RemoteException;
import java.util.logging.Level;

import edu.udo.cs.miningmart.exception.M4CompilerWarning;
import edu.udo.cs.miningmart.exception.UserError;

/**
 * This is the RMI interface class to send commands to the M4 compiler.
 * It is called to execute single steps, a sequence of steps or to
 * update statistics of the M4 model.
 * 
 * @author Martin Scholz
 * @version $Id: CompilerAccess.java,v 1.6 2006/04/11 14:10:09 euler Exp $
 */
public interface CompilerAccess {

    /**
     * This method is given the ID of the step to be compiled.
     * If the lazy mode flag is set, then a less time consuming
     * execution is performed.
     * <p>An exception is thrown if the step is not present in the
     * according M4 table, if parameters are missing, if references
     * are invalid, if the database connection fails or if the
     * execution of an operator fails.
     *
     * @param stepID    the ID of the step in <i>STEP_T</i>
     * @param lazyMode  a flag indicating if the compiler
     *                  should run in lazy mode or not
     */
    void compileStep(long stepID, boolean lazyMode)
    	throws M4CompilerInterfaceError, M4CompilerWarning, UserError;

	/*
	 * @see CompilerAccess#compileStep(long, boolean)
	 * 
     * @param stepID    the ID of the step in <i>STEP_T</i>
     * @param lazyMode  a flag indicating if the compiler
     *                  should run in lazy mode or not
     * @param verbosity the verbosity level for logfiles
     *
     * void compileStep(long stepID, boolean lazyMode, int verbosity)
     * throws RemoteException, M4CompilerInterfaceError;
     */

    /**
     * This method is given the name of the step to be compiled,
     * as specified in table <i>STEP_T</i>, attribute <i>ST_NAME</i>.
     * If the lazy mode flag is set, then a less time consuming
     * execution is performed.
     * <p>An exception is thrown if the step is not present in the
     * according M4 table, if parameters are missing, if references
     * are invalid, if the database connection fails or if the
     * execution of an operator fails.
     *
     * @param stepName  the name of the step in <i>STEP_T</i>
     * @param lazyMode  a flag indicating if the compiler
     *                  should run in lazy mode or not
     */
    void compileStep(String stepName, boolean lazyMode)
    	throws M4CompilerInterfaceError, M4CompilerWarning, UserError;

	/** 
     * This method is given the ID of a <code>Case</code> to be compiled.
     * The compiler calculates a valid sequential order of the embedded
     * <code>step</code>s with respect to the constraints introduced by
     * the <i>STEPSEQUENCE_T</i> table.
     * If the lazy mode flag is set, then a less time consuming execution
     * is performed.<p>
     * An exception is thrown if there is no valid sequence (e.g. cyclic
     * dependencies), if there are no steps connected to the given case,
     * if parameters are missing, if references are invalid, if the database
     * connection fails or if the execution of an operator fails.
     *
     * @param caseId      the ID of the case to be compiled
     * @param lazyMode    a flag indicating if the compiler
     *                    should run in lazy mode or not
	 * */
	void compileAll(long caseId, boolean lazyMode)
		throws M4CompilerInterfaceError, M4CompilerWarning, UserError;


	/*
	 * @see CompilerAccess#compileAll(long, boolean)
	 * 
     * @param caseId      the ID of the case to be compiled
     * @param lazyMode    a flag indicating if the compiler
     *                    should run in lazy mode or not
     * @param verbosity   the verbosity level for logfiles
	 * 
	void compileAll(long caseId, boolean lazyMode, int verbosity)
		throws RemoteException, M4CompilerInterfaceError;
	*/

	/** 
     * This method is given the ID of the first <code>Step</code> of a 
     * subtree to be compiled.
     * The compiler calculates a valid sequential order of the subtree
     * defined by and with respect to the constraints introduced by
     * the <i>STEPSEQUENCE_T</i> table.
     * If the lazy mode flag is set, then a less time consuming execution
     * is performed.<p>
     * An exception is thrown if there is no valid sequence (e.g. cyclic
     * dependencies), if there are no steps connected to the given case,
     * if parameters are missing, if references are invalid, if the database
     * connection fails or if the execution of an operator fails.
     *
     * @param stepId      the ID of the first step to be compiled
     * @param lazyMode    a flag indicating if the compiler
     *                    should run in lazy mode or not
	 * */
    void compileStepFrom(long stepId, boolean lazyMode)
		throws M4CompilerInterfaceError, M4CompilerWarning, UserError;

	/*
	 * @see CompilerAccess#compileStepFrom(long, boolean)
     *
     * @param stepId      the ID of the first step to be compiled
     * @param lazyMode    a flag indicating if the compiler
     *                    should run in lazy mode or not
     * @param verbosity   the verbosity level for logfiles
	 *
     * void compileStepFrom(long stepId, boolean lazyMode, int verbosity)
	 * throws RemoteException, M4CompilerInterfaceError;
	 */

	/**
	 * This method compiles all Steps that need to be compiled
	 * before the given Step can be compiled; then it compiles
	 * the given Step.
	 * 
	 * @param stepId the ID of the last step to be compiled
     * @param lazyMode a flag indicating if the compiler
     * should run in lazy mode or not
     * @param ignoreCompiledStatus if TRUE, only the graph structure
     * determines which Steps are compiled. If FALSE, only those
     * Steps among the predecessors of the given Step are compiled 
     * that have not been compiled yet. In the latter case, only
     * Steps on the shortest paths from a compiled step to the given Step
     * are compiled, and no Steps that precede an already compiled Step
     * are compiled.
	 */
    public void compileStepTo(long stepId, boolean lazyMode, boolean ignoreCompiledStatus)
		throws M4CompilerInterfaceError, M4CompilerWarning, UserError;
		
    /**
     * Uses the stored procedures of the database to update
     * all the statistics which are related to the specified concept.
     * These are the statistics of the related column sets and of all
     * the columns contained.
     *
     * @param conceptID the ID of the concept in table
     *                  <i>CONCEPT_T</i> for which the statistics
     *                  shall be updated
     */
    void updateStatisticsForConcept(long conceptID)
		throws RemoteException, M4CompilerInterfaceError;

    /**
     * Uses the stored procedures of the database to update
     * the statistics of the specified column set and of all the
     * columns contained in the column set.
     *
     * @param columnsetID the ID of column set in table
     *                    <i>COLUMNSET_T</i> for which the statistics
     *                    shall be updated
     */
    void updateColumnsetStatistics(long columnsetID)
		throws RemoteException, M4CompilerInterfaceError;

    /**
     * Uses the stored procedures of the database to update
     * the statistics of the specified column.
     *
     * @param columnID the ID of column set in table
     *                 <i>COLUMN_T</i> for which the statistics
     *                 shall be updated
     */
    void updateColumnStatistics(long columnID)
		throws RemoteException, M4CompilerInterfaceError;


    /**
     * Runs the garbage collection for the specified <code>Step</code>
     * and all subsequent dependent <code>Step</code>s with respect to
     * the entries in the table <i>STEPSEQUENCE_T</i>.
     *
     * @param stepId the ID of the (first) <code>Step</code> to run the
     * garbage collection for.
     */
	void deleteTrashForStep(long stepId)
		throws RemoteException, M4CompilerInterfaceError;

    /**
     * Runs the garbage collection for all <code>Step</code>s of a
     * specified <code>Case</code>.
     * 
     * @param caseId of the <code>Case</code> to run the garbage
     * collection for.
     */
	void deleteTrashForCase(long caseId)
		throws RemoteException, M4CompilerInterfaceError;
       

    /**
     * Reads the status message of a running compiler server thread.
     * @param caseId the id of the case to receive log information for.
     *        Because of the case locking mechanism each case is never compiled more
     *        than once at the same time, so this is a unique Thread identifier.
     *        A caseId of 0 or lower is interpreted as the default Print class,
     *        used before having found the right case or in case of severe errors.
     * @param numberOfLines the numberOfLines to be returned
     * @return the last number of lines from the status logfile of the thread
     * 		   The first line holds the line number after the last line, so that
     *         using this number as the line number in the next call will return
     *         the log messages collected in between. 
     * */
	String getStatusMessage(long caseId, int numberOfLines)
	   	throws RemoteException, M4CompilerInterfaceError;


    /**
     * Reads the status message of a running compiler server thread.
     * A start line number is specified. The log messages starting from this line
     * number up to the end are returned.
     * @param caseId the id of the case to receive log information for.
     *        Because of the case locking mechanism each case is never compiled more
     *        than once at the same time, so this is a unique Thread identifier.
     *        A caseId of 0 or lower is interpreted as the default Print class,
     *        used before having found the right case or in case of severe errors.
     * @param lineNumber the first line of the logfile to be returned
     * @return the last lines from the status logfile of the thread, starting with
     *         the lineNumber specified as a parameter.
     * */
	String getStatusFromLine(long caseId, int lineNumber)
		throws RemoteException, M4CompilerInterfaceError;
	
	/**
	 * Change the verbosity level of the compiler log messages for the given case.
	 * 
	 * @param caseId the id of the case to receive log information for.
     *        Because of the case locking mechanism each case is never compiled more
     *        than once at the same time, so this is a unique Thread identifier.
     *        A caseId of 0 or lower is interpreted as the default Print class,
     *        used before having found the right case or in case of severe errors.
     * @param newVerbosity The new verbosity. Use one of the public constants of
     *        the class edu.udo.cs.miningmart.m4.utils.Print, if possible.
	 */
	void changeCompilerOutputVerbosity(long caseId, Level newVerbosity) throws RemoteException, M4CompilerInterfaceError;
	
	/**
	 * This method checks for a given <code>Step</code>, if it can currently
	 * be compiled. Validity aspects are not checked, but for all <code>Step</code>s
	 * the given <code>Step</code> depends on (regarding to <i>STEPSEQUENCE_T</i>)
	 * it is made sure that they have recently (after the last garbage collection)
	 * been compiled, so the outputs of these <code>Step</code>s are available in
	 * the database.
	 * 
	 * @param stepId the ID of the <code>Step</code>
	 * @return if all <code>Step</code>s this <code>Step</code> depends on have
	 * been compiled
	 * */
	boolean isReadyForCompilation(long stepId)
	   	throws RemoteException, M4CompilerInterfaceError;

	/**
	 * This method kills a running compiler thread. Threads are identified by the
	 * <code>Case</code> they are compiling. Nevertheless, for a few seconds after
	 * launching a <code>compileStep</code> command this method may not succeed in
	 * identifying the right Thread. For this reason, there is a return value.
	 * It is a good idea to send kill request until <code>true</code> is returned.
	 * A Thread is killed by a special exception, which is finally translated to
	 * an <code>M4CompilerInterfaceError</code>.
	 * 
	 * @param caseId The ID of the case for which the compiler thread should be killed.
	 * @return <code>true</code> iff a thread for the given case has been found and the
	 *         the kill flag has been set.
	 * */
	boolean killCompilerThread(long caseId)
		throws RemoteException;

}
/*
 * Historie
 * --------
 *
 * $Log: CompilerAccess.java,v $
 * Revision 1.6  2006/04/11 14:10:09  euler
 * Updated license text.
 *
 * Revision 1.5  2006/04/06 16:31:09  euler
 * Prepended license remark.
 *
 * Revision 1.4  2006/03/30 16:07:16  scholz
 * fixed author tags for release
 *
 * Revision 1.3  2006/03/23 11:13:44  euler
 * Improved exception handling.
 *
 * Revision 1.2  2006/03/19 21:13:46  scholz
 * UserErrors are propagated now, useful for the GUI
 *
 * Revision 1.1  2006/01/03 09:54:31  hakenjos
 * Initial version!
 *
 */
