CREATE OR REPLACE TRIGGER BASEATTRIB_DEL_AFT_STM_TRIG
 AFTER DELETE ON BASEATTRIB_T
DECLARE
	   v_baId NUMBER DEFAULT NULL;
	   v_numBA NUMBER DEFAULT NULL;
	   v_baConId NUMBER DEFAULT NULL;
	   v_baMcfId NUMBER DEFAULT NULL;
	   v_baIterator NUMBER DEFAULT 1;
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       BASEATTRIB_DEL_AFT_STM_TRIG
   PURPOSE:    To manage object consistency when a base attribute is deleted

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        26-02-2002  Olaf Rem         1. Created this trigger.
   1.1		  22-03-2002  "	   			   1. Adapted to handle the change that
   			  			  				   	  a BaseAttribute can be part of more
											  Concepts
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.
******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* SET MultiColumnFeature invalid, IF this BaseAttribute IS the last or forelast member OF it
	    (a MCFeature should have at least two BaseAttributes) */

	 FOR i IN 1..Glob_var_package.t_ba_id.COUNT LOOP
	 	 --get the id of the base attribute that was deleted and its MultiColumnFeature id
	 	 v_baId := Glob_var_package.t_ba_id(i);
		 v_baMcfId := Glob_var_package.t_ba_mcfid(i);

		 --count the number of base attributes that still belong to this MCFeature
		 BEGIN
		 	 IF v_baMcfId IS NOT NULL
			 THEN
        		 SELECT COUNT (ba.BA_MCFID)
        		 INTO v_numBA
        		 FROM BASEATTRIB_T ba
    			 WHERE ba.BA_MCFID = v_baMcfId;
			 END IF;

    		 EXCEPTION
    		 	 WHEN NO_DATA_FOUND
    			 THEN
    			 	 v_numBA := NULL;
		 END;

		 --if this is the last attribute of the MCFeature, set it invalid.
		 --as the base attribute is now deleted the number of remaining
		 --attributes for the MCFeature must be zero.
		 IF (v_numBA IS NOT NULL) AND (v_numBA <= 1)
		 THEN
		 	 UPDATE MCFEATURE_T mcf
			 SET mcf.MCF_VALID = 'NO'
			 WHERE mcf.MCF_ID = v_baMcfId;
		 END IF;

	 END LOOP;


	 /* reinitialize global variables */
	 Glob_var_package.g_ba_counter := 0;
	 Glob_var_package.t_ba_id.DELETE;
 	 Glob_var_package.t_ba_mcfid.DELETE;
	 Glob_var_package.g_ba_deleted := FALSE;

END BASEATTRIB_DEL_AFT_STM_TRIG;

/
CREATE OR REPLACE TRIGGER BASEATTRIB_DEL_BEF_ROW_TRIG
 BEFORE DELETE ON BASEATTRIB_T
FOR EACH ROW
DECLARE
	  v_numBaseAttributes	 NUMBER DEFAULT NULL;
	  v_paramStId		 NUMBER DEFAULT NULL;
	  v_triggersOn 		 NUMBER DEFAULT NULL;

	  CURSOR c_concepts IS
	  SELECT bc.BC_CONID
	  FROM BA_CONCEPT_T bc
	  WHERE bc.BC_BAID = :OLD.BA_ID;

	  CURSOR c_inputParam IS
	  SELECT par.PAR_ID, par.PAR_STID
	  FROM PARAMETER_T par
	  WHERE par.PAR_OBJID = :OLD.BA_ID
	  AND par.PAR_OBJTYPE = 'BA'
	  AND par.PAR_TYPE = 'IN';

	  CURSOR c_caseInput IS
	  SELECT cai.CAI_CAID caid, cai.CAI_OBJID objid
	  FROM CASEINPUT_T cai
	  WHERE cai.CAI_OBJID = :OLD.BA_ID
	  AND cai.CAI_OBJTYPE = 'BA';

      CURSOR c_caseAttribute IS
	  SELECT caa.CAA_CAID caaid, caa.CAA_OBJID objid
	  FROM CASEATTRIB_T caa
	  WHERE caa.CAA_OBJID = :OLD.BA_ID
	  AND caa.CAA_OBJTYPE = 'BA';

/******************************************************************************
   NAME:       BASEATTRIB_DEL_BEF_ROW_TRIG
   PURPOSE:    To manage object consistency when a base attribute is deleted

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        25-02-2002  Olaf Rem         1. Created this trigger.
   1.1		  22-03-2002  Olaf Rem		   1. Uses delete_documentation procedure
   			  			  	   			   2. Adapted to handle the change that
										   	  a BaseAttribute can be part of more
											  Concepts.
										   3. Adapted to handle the change in how
										      operators used.
   1.2        06-05-2002  Erik Darwinkel   1. Adapted to handle ba_column table.
   1.3        28-11-2002  Erik Darwinkel   1. Added concept deleted check.
   1.4        06-02-2003  Martin Scholz    Trigger-off-flag tested.
******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	  --Store the id of this BaseAttribute
	  Glob_var_package.g_ba_counter := Glob_var_package.g_ba_counter + 1;
	  Glob_var_package.t_ba_id(Glob_var_package.g_ba_counter) := :OLD.BA_ID;

	  --Set global variable (used by BACONCEPT trigger)
	  Glob_var_package.g_ba_deleted := TRUE;

	  /* DELETE documentation FOR this BaseAttribute */
	  Delete_documentation(:OLD.BA_ID, 'BA');

	  -- only execute this part if a Concept is not being deleted!
	  IF NOT (Glob_var_package.g_con_deleted)
	  THEN
	  	 /* SET Concept invalid, IF this BaseAttribute IS the last member OF it */
	     FOR r_concepts IN c_concepts LOOP
	 	    BEGIN
		       SELECT COUNT(bc.BC_BAID)
			   INTO v_numBaseAttributes
			   FROM BA_CONCEPT_T bc
			   WHERE bc.BC_CONID = r_concepts.BC_CONID;

			   IF (v_numBaseAttributes = 1)   --BaseAttribute has not been deleted yet
			   THEN
			      UPDATE CONCEPT_T con
				  SET con.CON_VALID = 'NO'
				  WHERE con.CON_ID = r_concepts.BC_CONID;
			   END IF;

			   EXCEPTION
			      WHEN NO_DATA_FOUND THEN
				  NULL;
            END;
	     END LOOP;
	  END IF;
	  
	  --Remove all rows from BA_CONCEPT_T that contain this BaseAttribute
	  DELETE
	  FROM BA_CONCEPT_T bc
	  WHERE bc.BC_BAID = :OLD.BA_ID;

	  --Remove all rows from BA_COLUMN_T that contain this BaseAttribute
	  DELETE
	  FROM   ba_column_t bac
	  WHERE  bac.bac_baid = :OLD.BA_ID;

	 /* SET MultiColumnFeature invalid, IF this BaseAttribute IS the last member OF it

		This is handled further in a after statement trigger as it requires
		a select over the BASEATTRIB_T table which is not allowed in this trigger */

	  Glob_var_package.t_ba_mcfid(Glob_var_package.g_ba_counter) := :OLD.BA_MCFID;

	 /* SET corresponding Step invalid, WHERE this BaseAttribute IS an output parameter
	    (Assumption here is that the BaseAttribute can be an output parameter for one step only).*/

 	 v_paramStId := Find_step_by_output_object(:OLD.BA_ID, 'BA');

   	 IF v_paramStId IS NOT NULL
   	 THEN
   	 	 UPDATE STEP_T st
   		 SET st.ST_VALID = 'NO'
   		 WHERE st.ST_ID = v_paramStId;
   	 END IF;

	 /* SET ALL Steps invalid, which USE this BaseAttribute AS input parameter */
   	 FOR r_par IN c_inputParam LOOP
   	 	 IF r_par.PAR_STID IS NOT NULL
   		 THEN
   		 	 UPDATE STEP_T st
   			 SET st.ST_VALID = 'NO'
   			 WHERE st.ST_ID = r_par.PAR_STID;
   	 	 END IF;
   	 END LOOP;


	 /* SET CASE invalid, which USE this BaseAttribute AS caseInput (object-reference
	    FOR caseInput IS SET TO NULL, but ENTRY which defines a caseInput still EXISTS) */
	 FOR r_case IN c_caseInput LOOP
	 	 UPDATE CASE_T c
		 SET c.CA_VALID = 'NO'
		 WHERE c.CA_ID = r_case.caid;

		 /* NOTE: row is deleted from the CASEINPUT_T table instead
		    of setting the object reference to NULL. M4 does NOT allow TO have
		    OBJID set to NULL for this table and why would you want to have caseinputs that
		    point to objects that have been deleted? */
		 DELETE
		 FROM CASEINPUT_T cai
		 WHERE cai.CAI_OBJID = r_case.objid
		 AND cai.CAI_OBJTYPE = 'BA';
	 END LOOP;


	 /* SET CASE invalid, which USE this BaseAttribute AS targetAttribute (object-reference
	    FOR targetAttribute IS SET TO NULL, but ENTRY which defines a targetAttribute still EXISTS) */
	 FOR r_case IN c_caseAttribute LOOP
	 	 UPDATE CASE_T c
		 SET c.CA_VALID = 'NO'
		 WHERE c.CA_ID = r_case.caaid;

		 /* NOTE: row is deleted from the CASEATTRIB_T table instead
		    of setting the object reference to NULL. M4 does NOT allow TO have
		    CAID set to NULL for this table and why would you want to have caseattributes that
		    point to objects that have been deleted? */
		 DELETE
		 FROM CASEATTRIB_T caa
		 WHERE caa.CAA_OBJID = r_case.objid
		 AND caa.CAA_OBJTYPE = 'BA';
	 END LOOP;

	 /* Set parameter objid to NULL where this BaseAttribute is a parameter. */
	 UPDATE PARAMETER_T par
	 SET par.PAR_OBJID = NULL
	 WHERE par.PAR_OBJID = :OLD.BA_ID
	 AND par.PAR_OBJTYPE = 'BA';

END BASEATTRIB_DEL_BEF_ROW_TRIG;

/
CREATE OR REPLACE TRIGGER BASEATTRIB_INS_AFT_STM_TRIG
 AFTER INSERT ON BASEATTRIB_T
DECLARE
	   v_baId NUMBER DEFAULT NULL;
	   v_baMcfId NUMBER DEFAULT NULL;
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       BASEATTRIB_I_U_AFT_TRIG
   PURPOSE:    To manage object consistency when a BaseAttribute is inserted
   			   or updated.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        22-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

  	 /* SET MultiColumnFeature valid, IF this BaseAttribute IS the first
	 BaseAttribute OF the corresponding MultiColumnFeature (USE mcfeature-validity-FUNCTION)
	 (inserts only) */
	 FOR i IN 1..Glob_var_package.t_ba_id.COUNT LOOP
	 	 --get the id of multi column feature that corresponds to the base attribute that was inserted
		 v_baMcfId := Glob_var_package.t_ba_mcfid(i);

		 IF Is_mcfeature_valid(v_baMcfId)
		 THEN
		 	 UPDATE MCFEATURE_T mcf
			 SET mcf.MCF_VALID = 'YES'
			 WHERE mcf.MCF_ID = v_baMcfId;
		 ELSE
		 	 UPDATE MCFEATURE_T mcf
			 SET mcf.MCF_VALID = 'NO'
			 WHERE mcf.MCF_ID = v_baMcfId;
		 END IF;

	 END LOOP;

END BASEATTRIB_INS_AFT_STM_TRIG;

/
CREATE OR REPLACE TRIGGER BASEATTRIB_INS_BEF_ROW_TRIG
 BEFORE INSERT
 ON BASEATTRIB_T
FOR EACH ROW
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       BASEATTRIB_INS_BEF_ROW_TRIG
   PURPOSE:    To manage object consistency when a base attribute is inserted
   			   or updated.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        22-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN
	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* SET MultiColumnFeature valid, IF this BaseAttribute IS the first
	 BaseAttribute OF the corresponding MultiColumnFeature (USE mcfeature-validity-FUNCTION)
	 (inserts only)

	  Here only the relevant id's of the BaseAttribute that are inserted is saved. An after statement
	  trigger will then call the MCFeature validity function. This function does a select
	  over the BASEATTRIB_T table and this is not allowed in this trigger.
	 */
   	  Glob_var_package.g_ba_counter := Glob_var_package.g_ba_counter + 1;
   	  Glob_var_package.t_ba_id(Glob_var_package.g_ba_counter) := :NEW.BA_ID;
   	  Glob_var_package.t_ba_mcfid(Glob_var_package.g_ba_counter) := :NEW.BA_MCFID;

END BASEATTRIB_INS_BEF_ROW_TRIG;

/
CREATE OR REPLACE TRIGGER BASEATTRIB_INS_BEF_STM_TRIG
 BEFORE INSERT ON BASEATTRIB_T
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       BASEATTRIB_INS_BEF_STM_TRIG
   PURPOSE:    To initialize variables.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        20-09-2002  Erik Darwinkel   1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* initialize global variables */

	 Glob_var_package.g_ba_counter := 0;
	 Glob_var_package.t_ba_id.DELETE;
 	 Glob_var_package.t_ba_mcfid.DELETE;

END BASEATTRIB_INS_BEF_STM_TRIG;

/
CREATE OR REPLACE TRIGGER BASEATTRIB_UPD_BEF_ROW_TRIG
 BEFORE UPDATE OF BA_ATTRIBTYPE, BA_MCFID
 ON BASEATTRIB_T
FOR EACH ROW
DECLARE
	   v_hasConcept	 BOOLEAN   DEFAULT FALSE;
	   v_hasColumn	 BOOLEAN   DEFAULT FALSE;
	   v_hasStep	 BOOLEAN   DEFAULT FALSE;
	   v_paramStId	 NUMBER    DEFAULT NULL;
	   v_numSteps	 NUMBER    DEFAULT NULL;
	   v_numConcepts NUMBER    DEFAULT NULL;
	   v_numColumns  NUMBER    DEFAULT NULL;
	   v_triggersOn  NUMBER    DEFAULT NULL;

/******************************************************************************
   NAME:       BASEATTRIB_UPD_BEF_ROW_TRG
   PURPOSE:    SET this BaseAttribute valid, IF validity conditions are fulfilled,
	 		   otherwise SET this BaseAttribute invalid (inserts AND updates)

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        26-02-2002  Olaf Rem         1. Created this trigger.
   1.1		  22-03-2002  Olaf Rem		   1. Adapted to handle the change that
										   	  a BaseAttribute can be part of more
											  Concepts.
										   2. Now is only update trigger instead
										      of insert and update.
   1.2		  15-07-2002  Erik Darwinkel   1. Adapted to handle table ba_column_t.
   1.3        06-02-2003  Martin Scholz        Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* SET this BaseAttribute valid, IF validity conditions are fulfilled,
	 otherwise SET this BaseAttribute invalid (inserts AND updates)

	 A BaseAttribute is valid if:
	 ? it has a Concept (belongsToConcept has a value)
	 ? it has a corresponding Column, for type DB
	 ? a Step exists, where this BaseAttribute is an output parameter, for type MINING

	 */

	 --does it have a concept?
  	  BEGIN
	  	   SELECT COUNT(bc.BC_CONID)
		   INTO v_numConcepts
		   FROM BA_CONCEPT_T bc
		   WHERE bc.BC_BAID = :NEW.BA_ID;

		   IF (v_numConcepts > 0)
		   THEN
		       v_hasConcept := TRUE;
		   ELSE
		       v_hasConcept := FALSE;
		   END IF;

		   EXCEPTION
		       WHEN NO_DATA_FOUND
			   THEN
			       v_hasConcept := FALSE;
	  END;

	 --does it have a column if of type DB?
	 IF (:NEW.BA_ATTRIBTYPE = 'DB')
	 THEN
  	  BEGIN
	  	   SELECT COUNT(bac.BAC_ID)
		   INTO v_numColumns
		   FROM BA_COLUMN_T bac
		   WHERE bac.BAC_BAID = :NEW.BA_ID;

		   IF (v_numColumns > 0)
		   THEN
		       v_hasColumn := TRUE;
		   ELSE
		       v_hasColumn := FALSE;
		   END IF;

		   EXCEPTION
		       WHEN NO_DATA_FOUND
			   THEN
			       v_hasColumn := FALSE;
	  END;
	 END IF;

	 --does a Step exist where it is an output parameter if of type MINING?
	 IF (:NEW.BA_ATTRIBTYPE = 'MINING')
	 THEN

	 	 v_hasStep := Is_output_for_a_step(:NEW.BA_ID, 'BA');

		 --if this BaseAttribute is output of more than one Step, set it invalid
		 IF (v_hasStep IS NULL)
		 THEN
		 	 v_hasStep := FALSE;
		 END IF;
	 END IF;

	 -- now combine the gathered information and set this BaseAttribute valid or invalid
	 IF v_hasConcept AND (v_hasColumn OR v_hasStep)
	 THEN
	 	 :NEW.BA_VALID := 'YES';
	 ELSE
	 	 :NEW.BA_VALID := 'NO';
     END IF;

END BASEATTRIB_UPD_BEF_ROW_TRIG;

/
CREATE OR REPLACE TRIGGER BA_COLUMN_I_U_D_AFT_STM_TRIG
  AFTER INSERT OR UPDATE OR DELETE ON BA_COLUMN_T
DECLARE
	  v_triggersOn NUMBER DEFAULT NULL;

	  CURSOR c_baseattributes IS
	  SELECT ba.ba_id
	  FROM   baseattrib_t ba
	  ;

/******************************************************************************
   NAME:       BA_COLUMN_I_U_D_AFT_STM_TRIG
   PURPOSE:    Control low level object consistency after BA_COLUMN is changed.
   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        12-07-2002  Erik Darwinkel   1. Created this trigger.
   1.1        03-12-2002  Erik Darwinkel   2. Added column check.
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.
******************************************************************************/
BEGIN
	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 -- only execute this part if a BaseAttribute or Column is not being deleted!
	 IF NOT (Glob_var_package.g_ba_deleted) AND NOT (Glob_var_package.g_col_deleted)

	 THEN

		 FOR r_baseattribute IN c_baseattributes LOOP
		 	 IF is_baseattribute_valid(r_baseattribute.ba_id)
			 THEN
			 	 UPDATE baseattrib_t ba
				 SET    ba.ba_valid = 'YES'
				 WHERE  ba.ba_id = r_baseattribute.ba_id;
			 ELSE
		 	 	 UPDATE baseattrib_t ba
          	 	 SET    ba.ba_valid = 'NO'
	  	  	 	 WHERE  ba.ba_id = r_baseattribute.ba_id;
		 	 END IF;
	  	 END LOOP;

	 END IF;

END BA_COLUMN_DEL_AFT_STM_TRIG;

/
CREATE OR REPLACE TRIGGER BA_CONCEPT_DEL_AFT_STM_TRIG
 AFTER DELETE ON BA_CONCEPT_T
DECLARE

	   v_bcId 	 NUMBER DEFAULT NULL;
	   v_conId	 NUMBER DEFAULT NULL;
	   v_baId	 NUMBER DEFAULT NULL;

	   v_setInvalid BOOLEAN DEFAULT FALSE;
	   v_numConcepts NUMBER DEFAULT NULL;
	   v_numAttr 	 NUMBER DEFAULT NULL;
	   v_triggersOn  NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       BA_CONCEPT_DEL_AFT_STM_TRIG
   PURPOSE:    To manage object consistency when a link between a Concept and a
   			   BaseAttribute is deleted.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        25-03-2002  Olaf Rem         1. Created this trigger.
   1.1        29-04-2002  Erik Darwinkel   2. Made a few fixes.
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN
	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 -- only execute this trigger if a Concept or BaseAttribute is not being deleted!

		 FOR i IN 1..Glob_var_package.t_bc_id.COUNT LOOP

		 	 --get the id of the BaseAttribute and Concept
			 v_baId := Glob_var_package.t_bc_baid(i);
			 v_conId := Glob_var_package.t_bc_conid(i);

		 	 -- only execute this part if a Concept is not being deleted!
			 IF NOT (Glob_var_package.g_con_deleted)
			 THEN
			 	 /* SET Concept invalid IF this BaseAttribute IS the last member OF it */

				 v_setInvalid := FALSE;

				 BEGIN
				 	  SELECT COUNT(bc.BC_ID)
					  INTO v_numAttr
					  FROM BA_CONCEPT_T bc
					  WHERE bc.BC_CONID = v_conId;

					  IF (v_numAttr = 0)
					  THEN
					  	  v_setInvalid := TRUE;
					  END IF;

				 	  EXCEPTION
					      WHEN NO_DATA_FOUND THEN
						  v_setInvalid := TRUE;
				 END;

				 IF v_setInvalid
				 THEN
				 	 UPDATE CONCEPT_T con
					 SET con.CON_VALID = 'NO'
					 WHERE con.CON_ID = v_conId;
				 END IF;

			 END IF;

		 	 -- only execute this part if a BaseAttribute is not being deleted!
			 IF NOT (Glob_var_package.g_ba_deleted)
			 THEN

				 /* SET BaseAttribute invalid IF the Concept IS its last */

				 v_setInvalid := FALSE;

				 BEGIN
				 	  SELECT COUNT(bc.BC_ID)
					  INTO v_numConcepts
					  FROM BA_CONCEPT_T bc
					  WHERE bc.BC_BAID = v_baId;

					  IF (v_numConcepts = 0)
					  THEN
					  	  v_setInvalid := TRUE;
					  END IF;

				 	  EXCEPTION
					      WHEN NO_DATA_FOUND THEN
						  v_setInvalid := TRUE;
				 END;

				 IF v_setInvalid
				 THEN
				 	 UPDATE BASEATTRIB_T ba
					 SET ba.BA_VALID = 'NO'
					 WHERE ba.BA_ID = v_baId;
				 END IF;

			END IF;

		 END LOOP;


END BA_CONCEPT_DEL_AFT_STM_TRIG;

/
CREATE OR REPLACE TRIGGER BA_CONCEPT_DEL_BEF_ROW_TRIG
 BEFORE DELETE ON BA_CONCEPT_T
FOR EACH ROW
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       BA_CONCEPT_DEL_BEF_ROW_TRIG
   PURPOSE:    To manage object consistency when a link between BaseAttribute and
   			   Concept is deleted.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        25-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.
 
******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* SET Concept invalid IF this BaseAttribute IS the last member OF it */
	 
	 
	 /* SET BaseAttribute invalid IF the Concept IS its last */
	 
	 --store values for processing in an after statement trigger
	  Glob_var_package.g_bc_counter := Glob_var_package.g_bc_counter + 1;
	  Glob_var_package.t_bc_id(Glob_var_package.g_bc_counter) := :OLD.BC_ID;
	  Glob_var_package.t_bc_baid(Glob_var_package.g_bc_counter) := :OLD.BC_BAID;	  
	  Glob_var_package.t_bc_conid(Glob_var_package.g_bc_counter) := :OLD.BC_CONID;	 

END BA_CONCEPT_DEL_BEF_ROW_TRIG;

/ 

CREATE OR REPLACE TRIGGER BA_CONCEPT_I_AFT_STM_TRIG
 AFTER INSERT ON BA_CONCEPT_T
DECLARE
	   v_bcId NUMBER DEFAULT NULL;
	   v_bcConId NUMBER DEFAULT NULL;
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       BA_CONCEPT_I_AFT_STM_TRIG
   PURPOSE:    To manage object consistency when a BaseAttribute is inserted.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        25-03-2002  Olaf Rem         1. Created this trigger.
   1.1        20-09-2002  Erik Darwinkel   1. M4 model changes.
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* SET Concept valid, IF this BaseAttribute IS the first BaseAttribute
	  OF the corresponding Concept (USE concept-validity-FUNCTION) */

 	--Retrieve from the t_bc_id global PL/SQL table the BaseAttributes
	--that have been inserted AND SET the validity of the corresponding concept

	 FOR i IN 1..Glob_var_package.t_bc_id.COUNT LOOP
	 	 --get the id of the concept that corresponds to the base attribute that was inserted
		 v_bcConId := Glob_var_package.t_bc_conid(i);

		 IF Is_concept_valid(v_bcConId)
		 THEN
		 	 UPDATE CONCEPT_T con
			 SET con.CON_VALID = 'YES'
			 WHERE con.CON_ID = v_bcConId;
		 ELSE
		 	 UPDATE CONCEPT_T con
			 SET con.CON_VALID = 'NO'
			 WHERE con.CON_ID = v_bcConId;
		 END IF;

	 END LOOP;

END BA_CONCEPT_I_AFT_STM_TRIG;

/

CREATE OR REPLACE TRIGGER BA_CONCEPT_I_U_BEF_ROW_TRIG
 BEFORE INSERT OR UPDATE
 ON BA_CONCEPT_T
FOR EACH ROW
DECLARE
	   v_hasConcept	 BOOLEAN   DEFAULT FALSE;
	   v_numCols     NUMBER    DEFAULT NULL;
	   v_hasColumn	 BOOLEAN   DEFAULT FALSE;
	   v_hasStep	 BOOLEAN   DEFAULT FALSE;
	   v_paramStId	 NUMBER    DEFAULT NULL;
	   v_numSteps	 NUMBER    DEFAULT NULL;
	   v_attribType VARCHAR2(15) DEFAULT NULL;
	   v_triggersOn NUMBER       DEFAULT NULL;

/******************************************************************************
   NAME:       BA_CONCEPT_I_U_BEF_ROW_TRIG
   PURPOSE:    To manage object consistency when a base attribute is inserted
   			   or updated.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        22-03-2002  Olaf Rem         1. Created this trigger.
   1.1        22-04-2002  Erik Darwinkel   1. Changes for new table BA_COLUMN_T
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* SET this BaseAttribute valid, IF validity conditions are fulfilled,
	 otherwise SET this BaseAttribute invalid (inserts AND updates)

	 A BaseAttribute is valid if:
	 ? it has a Concept (belongsToConcept has a value)
	 ? it has a corresponding Column (correspondsToColumn has a value), for type DB
	 ? a Step exists, where this BaseAttribute is an output parameter, for type MINING

	 */

	 --does it have a concept?
 	 IF :NEW.BC_CONID IS NOT NULL
	 THEN
    	 v_hasConcept := TRUE;
	 END IF;

	 --does it have a column if of type DB?

	 BEGIN
	 	  SELECT COUNT(*)
		  INTO   v_numCols
		  FROM   BASEATTRIB_T ba
		  ,      BA_COLUMN_T bc
		  WHERE  ba.BA_ID = :NEW.BC_BAID
		  AND    ba.BA_ATTRIBTYPE = 'DB'
		  AND    ba.BA_ID = bc.BAC_BAID
		  ;

 		  IF (v_numCols > 0)
		  THEN
		 	 v_hasColumn := TRUE;
		  END IF;

    	  EXCEPTION
		     WHEN NO_DATA_FOUND THEN
			      v_hasColumn := FALSE;

	 END;

	 --does a Step exist where it is an output parameter if of type MINING?

 	 BEGIN
	 	  SELECT ba.BA_ATTRIBTYPE
		  INTO   v_attribType
		  FROM   BASEATTRIB_T ba
		  WHERE  ba.BA_ID = :NEW.BC_BAID;

    	  EXCEPTION
		     WHEN NO_DATA_FOUND THEN
			      v_attribType := '';

	 END;

	 IF (v_attribType = 'MINING')
	 THEN
	 	 v_hasStep := Is_output_for_a_step(:NEW.BC_BAID, 'BA');

		 --if this BaseAttribute is output of more than one Step, set it invalid
		 IF (v_hasStep IS NULL)
		 THEN
		 	 v_hasStep := FALSE;
		 END IF;
	 END IF;

	 --set the BaseAttribute valid or invalid
	 IF v_hasConcept AND (v_hasColumn OR v_hasStep)
	 THEN
	 	 UPDATE BASEATTRIB_T ba
		 SET ba.BA_VALID = 'YES'
		 WHERE ba.BA_ID = :NEW.BC_BAID;
	 ELSE
	 	 UPDATE BASEATTRIB_T ba
		 SET ba.BA_VALID = 'NO'
		 WHERE ba.BA_ID = :NEW.BC_BAID;
     END IF;


	 /* SET Concept valid, IF this BaseAttribute IS the first BaseAttribute
	  OF the corresponding Concept (USE concept-validity-FUNCTION)

	  Here only the relevant id's of the BaseAttribute that are inserted is saved. An after statement
	  trigger will then call the Concept validity function. This function does a select
	  over the BASEATTRIB_T table and this is not allowed in this trigger. */
	  
	 --store values for processing in an after statement trigger
	  Glob_var_package.g_bc_counter := Glob_var_package.g_bc_counter + 1;
	  Glob_var_package.t_bc_id(Glob_var_package.g_bc_counter) := :NEW.BC_ID;
	  Glob_var_package.t_bc_baid(Glob_var_package.g_bc_counter) := :NEW.BC_BAID;
	  Glob_var_package.t_bc_conid(Glob_var_package.g_bc_counter) := :NEW.BC_CONID;
	  
END BA_CONCEPT_I_U_BEF_ROW_TRIG;

/

CREATE OR REPLACE TRIGGER BA_CONCEPT_I_U_D_BEF_STM_TRIG
 BEFORE INSERT OR UPDATE OR DELETE ON BA_CONCEPT_T
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       BA_CONCEPT_I_U_D_BEF_STM_TRIG
   PURPOSE:    To initialize globals.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        20-09-2002  Erik Darwinkel   1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* Initialize global variables */

	 Glob_var_package.g_bc_counter := 0;
	 Glob_var_package.t_bc_id.DELETE;
	 Glob_var_package.t_bc_baid.DELETE;
 	 Glob_var_package.t_bc_conid.DELETE;

END BA_CONCEPT_I_U_D_BEF_STM_TRIG;

/
CREATE OR REPLACE TRIGGER CASEATTRIB_I_U_AFT_STM_TRIG
 AFTER INSERT OR UPDATE ON CASEATTRIB_T
DECLARE

	   v_caId NUMBER DEFAULT NULL;
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       CASEATTRIB_I_U_AFT_STM_TRIG
   PURPOSE:    To check the validity of a corresponding Case when a row
   			   is inserted or updated in this table.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        07-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.
******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 FOR i IN 1..Glob_var_package.t_caa_id.COUNT LOOP
	 	 /*get the id of the Case that corresponds to the CaseAttribute that
		   was inserted or updated */
		 v_caId := Glob_var_package.t_caa_caid(i);

		 IF (v_caId IS NOT NULL)
		 THEN
			 IF Is_case_valid(v_caId)
			 THEN
			 	 UPDATE CASE_T ca
				 SET ca.CA_VALID = 'YES'
				 WHERE ca.CA_ID = v_caId;
			 ELSE
			 	 UPDATE CASE_T ca
				 SET ca.CA_VALID = 'NO'
				 WHERE ca.CA_ID = v_caId;
			 END IF;
		 END IF;
	 END LOOP;

	 /* reinitialize global variables */

	 Glob_var_package.g_caa_counter := 0;
	 Glob_var_package.t_caa_id.DELETE;
	 Glob_var_package.t_caa_caid.DELETE;

END CASEATTRIB_I_U_AFT_STM_TRIG;

/
CREATE OR REPLACE TRIGGER CASEATTRIB_I_U_BEF_ROW_TRIG
 BEFORE INSERT OR UPDATE ON CASEATTRIB_T
FOR EACH ROW
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       CASEATTRIB_I_U_BEF_ROW_TRIG
   PURPOSE:    To check the validity of the corresponding Case when data is
   			   inserted or updated in this table.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        07-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.
******************************************************************************/
BEGIN
	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 Glob_var_package.g_caa_counter := Glob_var_package.g_caa_counter + 1;
	 Glob_var_package.t_caa_id(Glob_var_package.g_caa_counter) := :NEW.CAA_ID;
	 Glob_var_package.t_caa_caid(Glob_var_package.g_caa_counter) := :NEW.CAA_CAID;

END CASEATTRIB_I_U_BEF_ROW_TRIG;

/ 
CREATE OR REPLACE TRIGGER CASEINPUT_I_U_AFT_STM_TRIG
 AFTER INSERT OR UPDATE ON CASEINPUT_T
DECLARE

	   v_caId NUMBER DEFAULT NULL;
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       CASEINPUT_I_U_AFT_STM_TRIG
   PURPOSE:    To check the validity of a corresponding Case when a row
   			   is inserted or updated in this table.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        07-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.
******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;


	 FOR i IN 1..Glob_var_package.t_cai_id.COUNT LOOP
	 	 /*get the id of the Case that corresponds to the CaseInput that
		   was inserted or updated */
		 v_caId := Glob_var_package.t_cai_caid(i);

		 IF (v_caId IS NOT NULL)
		 THEN
			 IF Is_case_valid(v_caId)
			 THEN
			 	 UPDATE CASE_T ca
				 SET ca.CA_VALID = 'YES'
				 WHERE ca.CA_ID = v_caId;
			 ELSE
			 	 UPDATE CASE_T ca
				 SET ca.CA_VALID = 'NO'
				 WHERE ca.CA_ID = v_caId;
			 END IF;
		 END IF;
	 END LOOP;

	 /* reinitialize global variables */

	 Glob_var_package.g_cai_counter := 0;
	 Glob_var_package.t_cai_id.DELETE;
	 Glob_var_package.t_cai_caid.DELETE;

END CASEINPUT_I_U_AFT_STM_TRIG;

/
CREATE OR REPLACE TRIGGER CASEINPUT_I_U_BEF_ROW_TRIG
 BEFORE INSERT OR UPDATE ON CASEINPUT_T
FOR EACH ROW
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       CASEINPUT_I_U_BEF_ROW_TRIG
   PURPOSE:    To check the validity of the corresponding Case when data is
   			   inserted or updated in this table.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        07-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.
******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;


	 Glob_var_package.g_cai_counter := Glob_var_package.g_cai_counter + 1;
	 Glob_var_package.t_cai_id(Glob_var_package.g_cai_counter) := :NEW.CAI_ID;
	 Glob_var_package.t_cai_caid(Glob_var_package.g_cai_counter) := :NEW.CAI_CAID;

END CASEINPUT_I_U_BEF_ROW_TRIG;

/ 
CREATE OR REPLACE TRIGGER CASE_DEL_AFT_STM_TRIG
 AFTER DELETE ON CASE_T
DECLARE
	   v_caseId	 	   NUMBER DEFAULT NULL;
	   v_triggersOn            NUMBER DEFAULT NULL;
/******************************************************************************
   NAME:       CASE_DEL_AFT_STM_TRIG
   PURPOSE:    To control object consistency as a case is deleted

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        07-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.
******************************************************************************/
BEGIN

	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;
	 
	 /* reinitialize global variables */
	 Glob_var_package.g_ca_counter := 0;
	 Glob_var_package.t_ca_id.DELETE;


END CASE_DEL_AFT_STM_TRIG;

/ 
CREATE OR REPLACE TRIGGER CASE_DEL_BEF_ROW_TRIG
 BEFORE DELETE ON CASE_T
FOR EACH ROW
DECLARE
	   v_caseId  NUMBER DEFAULT NULL;
	   v_triggersOn NUMBER DEFAULT NULL;
/******************************************************************************
   NAME:       CASE_DEL_BEF_ROW_TRIG
   PURPOSE:    To perform work as each row is inserted or updated.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        07-03-2002  Olaf Rem         1. Created this trigger.
   1.1		  28-03-2002  "	   			   1. Adapted it to the M4 changes.
   1.2		  17-09-2002  Erik Darwinkel   1. Adapted to the M4 changes.
   1.3        06-02-2003  Martin Scholz    Trigger-off-flag tested.
******************************************************************************/
BEGIN

	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;

	 /* Store all id's of steps that are deleted.
	    Handle some delete actions in an after statement trigger. */
	  v_caseId := :OLD.CA_ID;

	  Glob_var_package.g_ca_counter := Glob_var_package.g_ca_counter + 1;
	  Glob_var_package.t_ca_id(Glob_var_package.g_ca_counter) := v_caseId;

	 /* ? DELETE documentation OF this CASE  */
	 Delete_documentation(v_caseId, 'CA');

	 /* ? SET ALL included Steps invalid  */
	 UPDATE STEP_T st
	 SET st.ST_CAID = NULL, st.ST_VALID = 'NO'
	 WHERE st.ST_CAID = v_caseId;

	 /* ? DELETE the list OF caseInput  */
	 DELETE
	 FROM CASEINPUT_T cai
	 WHERE cai.CAI_CAID = v_caseId;

 	 /* ? DELETE the list OF targetAttributes  */
	 DELETE
	 FROM CASEATTRIB_T caa
	 WHERE caa.CAA_CAID = v_caseId;

	  /* Set Concept relations to this Case on NULL. */
	  UPDATE concept_t
	  SET	 con_caid = NULL
	  WHERE	 con_caid = v_caseId;

	  /* delete all included Chains */
	  DELETE
	  FROM CHAIN_T ch
	  WHERE ch.CH_CASEID = v_caseId;


END CASE_DEL_BEF_ROW_TRIG;


/
CREATE OR REPLACE TRIGGER CASE_INS_UPD_BEF_ROW_TRIG
 BEFORE INSERT OR UPDATE OF CA_POPULATION, CA_OUTPUT
 ON CASE_T
FOR EACH ROW
DECLARE

	   v_hasStep  		 	 BOOLEAN DEFAULT FALSE;
	   v_hasValidCaInput	 BOOLEAN DEFAULT FALSE;
	   v_hasValidTargetAttr  BOOLEAN DEFAULT FALSE;
	   v_hasCaseOutput   	 BOOLEAN DEFAULT FALSE;
	   v_hasValidPopulation	 BOOLEAN DEFAULT FALSE;

	   v_caseId	  	  NUMBER DEFAULT NULL;
	   v_casePopId    NUMBER DEFAULT NULL;
	   v_caseOutputId NUMBER DEFAULT NULL;

	   v_numSteps	  	  NUMBER DEFAULT NULL;	   
	   v_numCaseInputs	  NUMBER DEFAULT NULL;

	   v_triggersOn NUMBER DEFAULT NULL;

	   CURSOR c_caInputs IS
	   SELECT cai.CAI_OBJID, cai.CAI_OBJTYPE
	   FROM CASEINPUT_T cai
	   WHERE cai.CAI_CAID = v_caseId;

	   CURSOR c_caTargets IS
	   SELECT caa.CAA_OBJID, caa.CAA_OBJTYPE
	   FROM CASEATTRIB_T caa
	   WHERE caa.CAA_CAID = v_caseId;


/******************************************************************************
   NAME: 	   CASE_INS_UPD_BEF_ROW_TRIG
   PURPOSE:    To set a Case valid or invalid when it is inserted or updated.

   			   A Case is valid, if
			    at least one Step exists
			    all objects of caseInput, if defined, must exist (object-references
			     must exist for all existing entries)
			    all BaseAttributes for targetAttributes, if defined, must exist
			    a Concept for caseOutput exists
			    a population exists which is member of caseInput


   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        07-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;


	 v_caseId := :NEW.CA_ID;

	 /* Store all id's of steps that are inserted or updated.
	    Handle some check actions in an after statement trigger. */

  	 Glob_var_package.g_ca_counter := Glob_var_package.g_ca_counter + 1;
  	 Glob_var_package.t_ca_id(Glob_var_package.g_ca_counter) := v_caseId;


	 /* Check if AT LEAST one Step EXISTS */
	 BEGIN
	 	  SELECT COUNT(st.ST_ID)
		  INTO v_numSteps
		  FROM STEP_T st
		  WHERE st.ST_CAID = v_caseId;

		  IF (v_numSteps > 0)
		  THEN
		  	  v_hasStep := TRUE;
		  ELSE
		  	  v_hasStep := FALSE;
		  END IF;

		  EXCEPTION
		      WHEN NO_DATA_FOUND THEN
			      v_hasStep := FALSE;
	 END;

	 /* Check if ALL objects OF caseInput, IF defined, exist (object-references
	    must exist FOR ALL existing entries) */
		v_hasValidCaInput := TRUE;
		FOR r_caInput IN c_caInputs LOOP
			IF (r_caInput.CAI_OBJID IS NULL) OR (r_caInput.CAI_OBJTYPE IS NULL)
			THEN
				v_hasValidCaInput := FALSE;
			END IF;
		END LOOP;

	 /* Check if ALL BaseAttributes FOR targetAttributes, IF defined, exist */
		v_hasValidTargetAttr := TRUE;
		FOR r_caTarget IN c_caTargets LOOP
			IF (r_caTarget.CAA_OBJID IS NULL) OR (r_caTarget.CAA_OBJTYPE IS NULL)
			THEN
				v_hasValidTargetAttr := FALSE;
			END IF;
		END LOOP;

	 /* Check if a Concept FOR caseOutput EXISTS */
	 IF :NEW.CA_OUTPUT IS NOT NULL
	 THEN
	 	 v_hasCaseOutput := TRUE;
	 END IF;

	 /* Check if a population EXISTS which IS member OF caseInput */
	 IF (:NEW.CA_POPULATION IS NOT NULL)
	 THEN
	 	 BEGIN
		 	  SELECT COUNT(cai.CAI_ID)
			  INTO v_numCaseInputs
			  FROM CASEINPUT_T cai
			  WHERE cai.CAI_OBJID = :NEW.CA_POPULATION
			  AND cai.CAI_OBJTYPE = 'CON';

			  IF v_numCaseInputs = 1
			  THEN
			  	  v_hasValidPopulation := TRUE;
			  END IF;

			  EXCEPTION
			      WHEN NO_DATA_FOUND THEN
				 	 v_hasValidPopulation := FALSE;
		 END;
	 END IF;

	 /* Set the Case validity using the gathered information */
	 IF v_hasStep AND v_hasValidCaInput	AND v_hasValidTargetAttr AND v_hasCaseOutput AND v_hasValidPopulation
	 THEN
	 	 :NEW.CA_VALID := 'YES';
	 ELSE
	     :NEW.CA_VALID := 'NO';
	 END IF;

END CASE_INS_UPD_BEF_ROW_TRIG;

/
CREATE OR REPLACE TRIGGER CASE_I_U_AFT_STM_TRIG
 AFTER INSERT OR UPDATE ON CASE_T
DECLARE

	   v_caseId	 NUMBER DEFAULT NULL;
	   v_triggersOn  NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       CASE_I_U_AFT_STM_TRIG
   PURPOSE:    To manage low level object consistency

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        07-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

/*	 FOR i IN 1..Glob_var_package.t_ca_id.COUNT LOOP
	 	 v_caseId := Glob_var_package.t_ca_id(i);

	 END LOOP;
*/

	 /* reinitialize global variables */
	 Glob_var_package.g_ca_counter := 0;
	 Glob_var_package.t_ca_id.DELETE;


END CASE_I_U_AFT_STM_TRIG;

/ 
CREATE OR REPLACE TRIGGER CHAIN_DEL_BEF_ROW_TRIG
 BEFORE DELETE ON CHAIN_T
FOR EACH ROW
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       CHAIN_DEL_BEF_ROW_TRIG
   PURPOSE:    To manage object consistency when a Chain is deleted

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        28-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN
	
	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;
 
	 /* Set object reference of Steps refering to this Chain to NULL */
	 UPDATE STEP_T st
	 SET st.ST_CHID = NULL
	 WHERE st.ST_CHID = :OLD.CH_ID;
	 
END CHAIN_DEL_BEF_ROW_TRIG;

/ 
CREATE OR REPLACE TRIGGER COLUMNSET_DEL_AFT_STM_TRIG
 AFTER DELETE ON COLUMNSET_T
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

	  CURSOR c_concepts IS
	  SELECT c.con_id
	  FROM   concept_t c
	  ,      columnset_t cs
	  WHERE  c.con_id = cs.cs_conid (+)
	  AND    cs.cs_conid IS NULL
	  AND    c.con_type = 'DB'
	  ;

/******************************************************************************
   NAME:       COLUMNSET_DEL_AFT_STM_TRIG
   PURPOSE:    To control object consistency after a ColumnSet is deleted
   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        03-05-2002  Erik Darwinkel   1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN
	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;

	 -- only execute this part if a Concept is not being deleted!
	 IF NOT (Glob_var_package.g_con_deleted)
	 THEN

	 	 /* SET Concept invalid if it is of type DB and
	 	    IF there is no ColumnSet for the Concept */

		 FOR r_concepts IN c_concepts LOOP
	 	 	 BEGIN
		 	  UPDATE concept_t con
			  SET    con.con_valid = 'NO'
			  WHERE  con.con_id = r_concepts.con_id;
		 	 END;
	  	 END LOOP;

	 END IF;

END COLUMNSET_DEL_AFT_STM_TRIG;

/
CREATE OR REPLACE TRIGGER COLUMNSET_DEL_BEF_ROW_TRIG
 BEFORE DELETE ON COLUMNSET_T
FOR EACH ROW
DECLARE
	   v_dropView BOOLEAN DEFAULT FALSE;
	   v_triggersOn NUMBER DEFAULT NULL;

	  CURSOR c_cols IS
	  SELECT col.COL_ID
	  FROM COLUMN_T col
	  WHERE col.COL_CSID = :OLD.CS_ID;

/******************************************************************************
   NAME:       COLUMNSET_DEL_BEF_ROW_TRIG
   PURPOSE:    To control object consistency when a ColumnSet is deleted
   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        19-02-2002  Olaf Rem         1. Created this trigger.
   1.1        19-03-2002  Olaf Rem         1. Added procedure for deleting
   			  			  	   			      documentation.
										   2. Delete statistics changed
   1.2        03-05-2002  Erik Darwinkel   1. Adapted to M4 changes.
   1.3        21-11-2002  Erik Darwinkel   1. Delete columns.
   1.4        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN
	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;

	 /* delete documentation for this ColumnSet */
	 Delete_documentation(:OLD.CS_ID, 'CS');

    /* delete statistics for all included Columns */
	FOR r_cols IN c_cols LOOP

	 DELETE COLSTATIST1_T colst1
	 WHERE colst1.COLST1_COLID = r_cols.COL_ID;

	 DELETE COLSTATIST2_T colst2
	 WHERE colst2.COLST2_COLID = r_cols.COL_ID;

	END LOOP;

	 /* Delete Columns for this ColumnSet */
	DELETE COLUMN_T col
	WHERE col.COL_CSID = :OLD.CS_ID;

	 /* for all KeyHeads which use this ColumnSet, set attribute primaryKey or foreignKey to NULL */
	 UPDATE KEYHEAD_T kh
	 SET kh.KH_PKCSID = NULL
	 WHERE kh.KH_PKCSID = :OLD.CS_ID;

	 UPDATE KEYHEAD_T kh
	 SET kh.KH_FKCSID = NULL
	 WHERE kh.KH_FKCSID = :OLD.CS_ID;

	 /* set Relations invalid, which use this ColumnSet (in attribute correspondsToColumnSet) */
	 UPDATE RELATION_T rel
	 SET rel.REL_VALID = 'NO',  rel.REL_CSID = NULL
	 WHERE rel.REL_CSID = :OLD.CS_ID;

	 /* delete statistics for this ColumnSet */
	 DELETE CSSTATIST_T csst
	 WHERE csst.CSST_CSID = :OLD.CS_ID;

	 /* delete actual DB-View-object for this Concept, if it is of type MINING
	 How do I do this???

	 v_dropView := FALSE;

	 -- CHECK IF corresponding Concept EXISTS AND IS OF TYPE MINING
	 BEGIN
	 	  SELECT con.CON_TYPE
		  INTO v_conType
		  FROM CONCEPT_T con
		  WHERE con.CON_CSID = :OLD.CS_ID;

		  IF v_conType = 'MINING'
		  THEN
		  	  v_dropView := TRUE;
		  END IF;

		  EXCEPTION
		  	  WHEN NO_DATA_FOUND THEN
			      NULL;
			  WHEN TOO_MANY_ROWS THEN
			      NULL;
	 END;

	 -- implement simple local delete
	 IF (:OLD.CS_NAME IS NOT NULL) AND v_dropView
	 THEN
	 	 DROP VIEW :OLD.CS_NAME;   	   			 -- doesn't work: DROP not allowed in PL/SQL (?)
	 END IF;

	 */
END COLUMNSET_DEL_BEF_ROW_TRIG;

/

CREATE OR REPLACE TRIGGER COLUMNSET_I_AFT_STM_TRIG
 AFTER INSERT ON COLUMNSET_T
DECLARE
	   v_csId NUMBER DEFAULT NULL;
	   v_csConId NUMBER DEFAULT NULL;
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       COLUMNSET_I_AFT_STM_TRIG
   PURPOSE:    To manage object consistency when a ColumnSet is inserted.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        30-09-2002  Erik Darwinkel   1. Created trigger
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;

	 /* SET Concept validity, (USE concept-validity-FUNCTION) */

 	--Retrieve from the t_cs_id global PL/SQL table the ColumnSets
	--that have been inserted AND SET the validity of the corresponding concept

	 FOR i IN 1..Glob_var_package.t_cs_id.COUNT LOOP
	 	 --get the id of the concept that corresponds to the ColumnSet that was inserted
		 v_csConId := Glob_var_package.t_cs_conid(i);

		 IF Is_concept_valid(v_csConId)
		 THEN
		 	 UPDATE CONCEPT_T con
			 SET con.CON_VALID = 'YES'
			 WHERE con.CON_ID = v_csConId;
		 ELSE
		 	 UPDATE CONCEPT_T con
			 SET con.CON_VALID = 'NO'
			 WHERE con.CON_ID = v_csConId;
		 END IF;

	 END LOOP;

END COLUMNSET_I_AFT_STM_TRIG;

/
CREATE OR REPLACE TRIGGER COLUMNSET_I_U_BEF_ROW_TRIG
 BEFORE INSERT OR UPDATE
 ON COLUMNSET_T
FOR EACH ROW
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       COLUMNSET_I_U_BEF_ROW_TRIG
   PURPOSE:    To manage object consistency when a columnset is inserted
   			   or updated.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        30-09-2002  Erik Darwinkel   1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;

	 /* Here only the id of the Concept belonging to the ColumnSet that is inserted is saved.
	  An after statement trigger will then call the Concept validity function. This function
	  does a select over the COLUMNSET_T table and this is not allowed in this trigger. */

	 --store values for processing in an after statement trigger
	  Glob_var_package.g_cs_counter := Glob_var_package.g_cs_counter + 1;
	  Glob_var_package.t_cs_id(Glob_var_package.g_cs_counter) := :NEW.CS_ID;
	  Glob_var_package.t_cs_conid(Glob_var_package.g_cs_counter) := :NEW.CS_CONID;

END COLUMNSET_I_U_BEF_ROW_TRIG;

/
CREATE OR REPLACE TRIGGER COLUMNSET_I_U_D_BEF_STM_TRIG
 BEFORE INSERT OR UPDATE OR DELETE ON COLUMNSET_T
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       COLUMNSET_I_U_D_BEF_STM_TRIG
   PURPOSE:    To initialize globals.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        30-09-2002  Erik Darwinkel   1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;

	 /* Initialize global variables */

	 Glob_var_package.g_cs_counter := 0;
	 Glob_var_package.t_cs_id.DELETE;
 	 Glob_var_package.t_cs_conid.DELETE;

END COLUMNSET_I_U_D_BEF_STM_TRIG;

/
CREATE OR REPLACE TRIGGER COLUMN_DEL_BEF_ROW_TRIG
 BEFORE DELETE ON COLUMN_T
FOR EACH ROW
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       COLUMN_DEL_BEF_ROW_TRIG
   PURPOSE:    Control low level object consistency when DELETING from COLUMN.
   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        18-02-2002  Olaf Rem         1. Created this trigger.
   1.1  	  19-03-2002  Olaf Rem		   1. Changed delete documentation
   			  			  	   			   	  to use a procedure.
										   2. Changed deletion of statistics due
										   	  to changed settings in M4 Relational.
   1.2        07-05-2002  Erik Darwinkel   1. Adapted to changes in ba_column.
   1.3        03-12-2002  Erik Darwinkel   1. Update base attribute validation here
                                           for performance reasons.
   1.4        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;

	 Glob_var_package.g_col_deleted := TRUE;

	 /* Set BaseAttribute validation from type 'DB' to 'NO' for this column */
 	 UPDATE baseattrib_t ba
 	 SET    ba.ba_valid = 'NO'
 	 WHERE  ba.ba_attribtype = 'DB'
	 AND	ba.ba_id = (SELECT bc.bac_baid FROM ba_column_t bc WHERE  bc.bac_colid = :OLD.COL_ID);

	 /* delete BaseAttribute relation for this Column  */
	 DELETE FROM   ba_column_t bc
	 WHERE  bc.bac_colid = :OLD.COL_ID;

	 /* delete documentation for this Column */
	 delete_documentation(:OLD.COL_ID, 'COL');

	 /* for all KeyMembers which use this Column, set attribute
	 primaryKeyColumn or foreignKeyColumn to NULL */
	 UPDATE KEYMEMBER_T km
	 SET km.KM_PKCOLID = NULL
	 WHERE km.KM_PKCOLID = :OLD.COL_ID;

	 UPDATE KEYMEMBER_T km
	 SET km.KM_FKCOLID = NULL
	 WHERE km.KM_FKCOLID = :OLD.COL_ID;

	 /* delete statistics for this Column  */
	 DELETE COLSTATIST1_T colst1
	 WHERE colst1.COLST1_COLID = :OLD.COL_ID;

	 DELETE COLSTATIST2_T colst2
	 WHERE colst2.COLST2_COLID = :OLD.COL_ID;

	 Glob_var_package.g_col_deleted := FALSE;

END COLUMN_DEL_BEF_ROW_TRIG;

/
CREATE OR REPLACE TRIGGER CONCEPT_DEL_AFT_STM_TRIG
 AFTER DELETE ON CONCEPT_T
DECLARE

	  v_conId  	  	NUMBER	DEFAULT NULL;
	  v_paramStId	NUMBER  DEFAULT NULL;
	  v_triggersOn NUMBER DEFAULT NULL;

	  CURSOR c_inputParam IS
	  SELECT par.PAR_ID, par.PAR_STID
	  FROM PARAMETER_T par
	  WHERE par.PAR_OBJID = v_conId
	  AND par.PAR_OBJTYPE = 'CON'
	  AND par.PAR_TYPE = 'IN';

	  CURSOR c_caseInput IS
	  SELECT cai.CAI_CAID caid, cai.CAI_OBJID objid
	  FROM CASEINPUT_T cai
	  WHERE cai.CAI_OBJID = v_conId
	  AND cai.CAI_OBJTYPE = 'CON';


/******************************************************************************
   NAME:       CONCEPT_DEL_AFT_STM_TRIG
   PURPOSE:    To manage object consistency when a concept is deleted.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        01-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-05-2002  Erik Darwinkel   1. Made fixes.
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;

	 /* ? DELETE documentation FOR this Concept */

	 FOR i IN 1..Glob_var_package.t_con_id.COUNT LOOP
	 	 v_conId := Glob_var_package.t_con_id(i);
		 Delete_documentation(v_conId, 'CON');
	 END LOOP;

	 /* ? SET Concept invalid, which uses this Concept AS restrictionToConcept
	 	  within a RoleRestriction (automatically through RoleRestriction-TRIGGER) */

	 FOR i IN 1..Glob_var_package.t_con_id.COUNT LOOP
	 	 v_conId := Glob_var_package.t_con_id(i);
		 UPDATE ROLERESTRICTION_T rr
		 SET rr.RR_TOCONID = NULL
		 WHERE rr.RR_TOCONID = v_conId;
	 END LOOP;


	 /* ? SET Step invalid, which uses this Concept AS output parameter */
	 FOR i IN 1..Glob_var_package.t_con_id.COUNT LOOP
	 	 v_conId := Glob_var_package.t_con_id(i);

 	 	 v_paramStId := Find_step_by_output_object(v_conId, 'CON');

	   	 IF v_paramStId IS NOT NULL
	   	 THEN
	   	 	 UPDATE STEP_T st
	   		 SET st.ST_VALID = 'NO'
	   		 WHERE st.ST_ID = v_paramStId;
	   	 END IF;
	 END LOOP;


	 /* ? SET ALL Steps invalid, which USE this Concept AS input parameter */

	 FOR i IN 1..Glob_var_package.t_con_id.COUNT LOOP
	 	 v_conId := Glob_var_package.t_con_id(i);
		 BEGIN
	    	 FOR r_par IN c_inputParam LOOP
	    	 	 IF r_par.PAR_STID IS NOT NULL
	    		 THEN
	    		 	 UPDATE STEP_T st
	    			 SET st.ST_VALID = 'NO'
	    			 WHERE st.ST_ID = r_par.PAR_STID;
	    	 	 END IF;
	    	 END LOOP;

			 EXCEPTION
			 	 WHEN NO_DATA_FOUND
				 THEN
				 NULL;
		 END;
	 END LOOP;

	 /* ? SET CASE invalid, which uses this Concept AS caseInput (object-reference
	 	  FOR caseInput IS SET TO NULL, but ENTRY which defines a caseInput still EXISTS) */
	 FOR r_case IN c_caseInput LOOP
	 	 UPDATE CASE_T c
		 SET c.CA_VALID = 'NO'
		 WHERE c.CA_ID = r_case.caid;

		 /* NOTE: row is deleted from the CASEINPUT_T table instead
		    of setting the object reference to NULL. M4 does NOT allow TO have
		    OBJID set to NULL for this table and why would you want to have caseinputs that
		    point to objects that have been deleted? */
		 DELETE
		 FROM CASEINPUT_T cai
		 WHERE cai.CAI_OBJID = r_case.objid
		 AND cai.CAI_OBJTYPE = 'CON';
	 END LOOP;

 	 Glob_var_package.g_con_deleted := FALSE;
	 
END CONCEPT_DEL_AFT_STM_TRIG;

/
CREATE OR REPLACE TRIGGER CONCEPT_DEL_BEF_ROW_TRIG
 BEFORE DELETE ON CONCEPT_T
FOR EACH ROW
DECLARE

	  v_conId  	  	NUMBER	DEFAULT NULL;
	  v_numConcepts NUMBER  DEFAULT NULL;
	  v_triggersOn NUMBER DEFAULT NULL;


	  CURSOR c_attribs IS
	  SELECT bc.BC_BAID
	  FROM BA_CONCEPT_T bc
	  WHERE bc.BC_CONID = :OLD.CON_ID;

/******************************************************************************
   NAME:       CONCEPT_DEL_BEF_ROW_TRIG
   PURPOSE:    To manage object consistency when a Concept is deleted

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        01-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-05-2002  Erik Darwinkel   1. Delete ColumnSets as well.
                                           2. Update MCFeature.
   1.2        17-09-2002  Erik Darwinkel   1. Adapted to M4 changes.
   1.3        28-11-2002  Erik Darwinkel   1. Delete BaseAttributes, Relations
   1.4        06-02-2003  Martin Scholz    Trigger-off-flag tested.
   			  			  	   			      and Coordinates.
******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* Store all id's of concepts that are deleted.
	    Handle delete actions in an after statement trigger. */
	  v_conId := :OLD.CON_ID;

	  Glob_var_package.g_con_counter := Glob_var_package.g_con_counter + 1;
	  Glob_var_package.t_con_id(Glob_var_package.g_con_counter) := v_conId;

	 /* ? DELETE ALL isA-relations FOR this Concept */

	 DELETE
	 FROM CONCEPTISA_T cisa
	 WHERE cisa.CISA_SUPERCONID = v_conId
	 OR cisa.CISA_SUBCONID = v_conId;

	 /* ? DELETE the extensionally equivalent projection-relation FOR this
	 	  Concept */
	 DELETE
	 FROM PROJECTION_T pro
	 WHERE pro.PRO_FROMCONID = v_conId
	 OR pro.PRO_TOCONID = v_conId;

	 /* ? DELETE ALL existing RoleRestrictions FOR this Concept (which uses
	 	  this Concept AS restrictionForConcept) */

	 DELETE
	 FROM ROLERESTRICTION_T rr
	 WHERE rr.RR_FROMCONID = v_conId;

	 /* ? DELETE ALL ColumnSets for this Concept */

	 DELETE
	 FROM columnset_t cs
	 WHERE cs.cs_conid = v_conId;

	 /* Delete ALL Relations, which USE this Concept */

	 DELETE RELATION_T rel
	 WHERE rel.REL_FROMCONID = v_conId;

	 DELETE RELATION_T rel
	 WHERE rel.REL_TOCONID = v_conId;

	 /* Delete ALL Coordinates, which USE this Concept */

	 DELETE HCI_COORD_T co
	 WHERE co.obj_id = v_conId;

	 /* ? SET ALL MultiColumnFeatures invalid, which USE this Concept */
	 UPDATE MCFEATURE_T mcf
	 SET mcf.MCF_CONID = NULL, mcf.MCF_VALID = 'NO'
	 WHERE mcf.MCF_CONID = v_conId;

	 /* ? SET CASE invalid, which uses this Concept AS caseOutput (caseOutput IS
	 	  SET TO NULL) OR population (population IS SET TO NULL) */
	 UPDATE CASE_T ca
	 SET ca.CA_OUTPUT = NULL, ca.CA_VALID = 'NO'
	 WHERE ca.CA_OUTPUT = v_conId;

	 UPDATE CASE_T ca
	 SET ca.CA_POPULATION = NULL, ca.CA_VALID = 'NO'
	 WHERE ca.CA_POPULATION = v_conId;

	 /* Delete ALL included BaseAttributes if they do not belong to any other Concept */
	 FOR r_attribs IN c_attribs LOOP
	 	 BEGIN
		 	  SELECT COUNT(bc.BC_CONID)
			  INTO v_numConcepts
			  FROM BA_CONCEPT_T bc
			  WHERE bc.BC_BAID = r_attribs.BC_BAID;

			  IF (v_numConcepts = 1)   --Concept has not been deleted yet
			  THEN
			 	 DELETE
			 	 FROM  BASEATTRIB_T ba
			 	 WHERE ba.BA_ID = r_attribs.BC_BAID;
			  END IF;

			  EXCEPTION
			      WHEN NO_DATA_FOUND THEN
				  NULL;
		 END;
	 END LOOP;

	  /* Remove all rows from BA_CONCEPT_T that contain this Concept */
	  DELETE
	  FROM BA_CONCEPT_T bc
	  WHERE bc.BC_CONID = :OLD.CON_ID;


END CONCEPT_DEL_BEF_ROW_TRIG;

/
CREATE OR REPLACE TRIGGER CONCEPT_DEL_BEF_STM_TRIG
 BEFORE DELETE ON CONCEPT_T
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       CONCEPT_DEL_BEF_STM_TRIG
   PURPOSE:    To do initializations before a concept is deleted.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        07-05-2002  Erik Darwinkel   1. Create trigger.
   1.1        28-11-2002  Erik Darwinkel   1. Added g_con_deleted.
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* initialize global variables */
	 Glob_var_package.g_con_counter := 0;
	 Glob_var_package.t_con_id.DELETE;
	 Glob_var_package.g_con_deleted := TRUE;

END CONCEPT_DEL_BEF_STM_TRIG;

/
CREATE OR REPLACE TRIGGER CONCEPT_INS_UPD_BEF_ROW_TRIG
 BEFORE INSERT OR UPDATE OF CON_TYPE
 ON CONCEPT_T
FOR EACH ROW
DECLARE

   v_hasColumnSet BOOLEAN		 DEFAULT FALSE;
   v_numColumnSets		  NUMBER		 DEFAULT NULL;
   v_hasStep 	  BOOLEAN		 DEFAULT FALSE;
   v_hasBaseAttribute 	  BOOLEAN		 DEFAULT FALSE;
   v_hasRoleRes   BOOLEAN		 DEFAULT FALSE;
   v_roleResValid BOOLEAN		 DEFAULT FALSE;
   v_conId		  NUMBER		 DEFAULT NULL;
   v_conType	  VARCHAR2(10)	 DEFAULT NULL;
   v_conValid	  BOOLEAN		 DEFAULT FALSE;
   v_stId		  NUMBER		 DEFAULT NULL;
   v_stDummy	  NUMBER		 DEFAULT NULL;
   v_numAttr	  NUMBER		 DEFAULT NULL;
   v_triggersOn   NUMBER                 DEFAULT NULL;


   CURSOR c_roleRes IS
   SELECT rr.RR_FROMCONID, rr.RR_TOCONID, rr.RR_RELID
   FROM ROLERESTRICTION_T rr
   WHERE rr.RR_FROMCONID = v_conId;


/******************************************************************************
   NAME:       CONCEPT_
   PURPOSE:    To set this Concept valid, if validity conditions are fulfilled,
   			   otherwise set this Concept invalid.

			   A Concept is valid, if
			   ? it has a corresponding ColumnSet for type DB
			   ? a Step exists, where this Concept is an output parameter,
			     for type MINING
			   ? at least one included BaseAttribute exists
			   ? if RoleRestrictions exist for this Concept (attribute
			     restrictionForConcept), then they must have a value for
				 restrictionToConcept and restrictionForRelation


   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        28-02-2002  Olaf Rem         1. Created this trigger.
   1.1		  25-03-2002  Olaf Rem		   1. Adapted it to the changed M4
   1.2        15-07-2002  Erik Darwinkel   1. Adapted it to the changed M4
   1.3        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* Get the concept id and type if available */
	 v_conId := :NEW.CON_ID;
	 v_conType := :NEW.CON_TYPE;

	 /* Check if the Concept has a ColumnSet if the Concept is of type DB */
	 IF (v_conType = 'DB')
	 THEN
  	  BEGIN
	  	   SELECT COUNT(cst.CS_ID)
		   INTO v_numColumnSets
		   FROM COLUMNSET_T cst
		   WHERE cst.CS_CONID = v_conId;

		   IF (v_numColumnSets > 0)
		   THEN
		       v_hasColumnSet := TRUE;
		   ELSE
		       v_hasColumnSet := FALSE;
		   END IF;

		   EXCEPTION
		       WHEN NO_DATA_FOUND
			   THEN
			       v_hasColumnSet := FALSE;
	  END;
	 END IF;

	 /* Check if a Step exists, where this Concept is an output parameter, for type MINING */
	 IF v_conType = 'MINING'
	 THEN
	 	 v_hasStep := Is_output_for_a_step(v_conId, 'CON');

		 --if this BaseAttribute is output of more than one Step, set it invalid
		 IF (v_hasStep IS NULL)
		 THEN
		 	 v_hasStep := FALSE;
		 END IF;
	 END IF;

 	 /* Check if at least one included BaseAttribute exists */
	 BEGIN
    	 SELECT COUNT(bc.BC_BAID)
    	 INTO 	v_numAttr
    	 FROM 	BA_CONCEPT_T  bc
    	 WHERE 	bc.BC_CONID = v_conId;

		 IF v_numAttr > 0
		 THEN
		 	 v_hasBaseAttribute := TRUE;
		 ELSE
		 	 v_hasBaseAttribute := FALSE;	-- if no BaseAttribute exists the Concept is invalid
		 END IF;

	 EXCEPTION
	     WHEN NO_DATA_FOUND THEN
		 	  v_hasBaseAttribute := FALSE; 	-- if no BaseAttribute exists the Concept is invalid
	 END;

	 /* If RoleRestrictions exist for this Concept (attribute restrictionForConcept),
		then they must have a value for restrictionToConcept and restrictionForRelation */
	 v_roleResValid := TRUE;
	 FOR r_role IN c_roleRes LOOP
	 	 v_hasRoleRes := TRUE;
		 IF (r_role.RR_TOCONID IS NULL) OR (r_role.RR_RELID IS NULL)
		 THEN
		 	 v_roleResValid := FALSE;
		 END IF;
	 END LOOP;

	 /* Now set the validity of the Concept using the information gathered previously */
	 IF (v_hasColumnSet OR v_hasStep) AND v_hasBaseAttribute AND (NOT v_hasRoleRes)
	 THEN
	 	 v_conValid := TRUE;
	 END IF;

	 IF (v_hasColumnSet OR v_hasStep) AND v_hasBaseAttribute AND v_hasRoleRes AND v_roleResValid
	 THEN
	 	 v_conValid := TRUE;
	 END IF;

	 IF v_conValid
	 THEN
	 	 :NEW.CON_VALID := 'YES';
	 ELSE
	 	 :NEW.CON_VALID := 'NO';
	 END IF;

END CONCEPT_INS_UPD_BEF_ROW_TRIG;

/
CREATE OR REPLACE TRIGGER KEYHEAD_DEL_BEF_ROW_TRIG
 BEFORE DELETE ON KEYHEAD_T
FOR EACH ROW
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       KEYHEAD_DEL_BEF_ROW_TRIG
   PURPOSE:    To control object consistency when a key is deleted
   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        19-02-2002  Olaf Rem         1. Created this trigger.
   1.1		  19-03-2002  Olaf Rem		   1. Changed deletion of documentation
   			  			  	   			      to use a procedure.
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* delete documentation for this Key */
	 Delete_documentation(:OLD.KH_ID, 'KH');

	 /* delete all KeyMembers of this Key/KeyHead */
	 DELETE KEYMEMBER_T km
	 WHERE km.KM_KHID = :OLD.KH_ID;

	 /* set Relations invalid, which use this Key (in attributes correspondsToForeignKey) */
	 UPDATE RELATION_T rel
	 SET rel.REL_FROMKID = NULL, rel.REL_VALID = 'NO'
	 WHERE rel.REL_FROMKID = :OLD.KH_ID;

	 UPDATE RELATION_T rel
	 SET rel.REL_TOKID = NULL, rel.REL_VALID = 'NO'
	 WHERE rel.REL_TOKID = :OLD.KH_ID;

END KEYHEAD_DEL_BEF_ROW_TRIG;

/
CREATE OR REPLACE TRIGGER MCFEATURE_DEL_BEF_ROW_TRIG
 BEFORE DELETE ON MCFEATURE_T
FOR EACH ROW
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       MCFEATURE_DEL_BEF_ROW_TRIG
   PURPOSE:    To manager object consistency when a multi column feature
   			   is deleted.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        22-02-2002  Olaf Rem         1. Created this trigger.
   1.1		  21-03-2002  Olaf Rem		   1. Changed deletion of documentation
   			  			  	   			   	  now using a procedure.
										   2. Changed setting references from 
										      BaseAttributes to MultiColumnFeature
											  to NULL due to changes in M4 Relational.
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* DELETE documentation FOR this MultiColumnFeature */
	 delete_documentation(:OLD.MCF_ID, 'MCF');

	 /* SET object-reference OF this MultiColumnFeature FOR ALL member BaseAttributes TO NULL */
	 UPDATE baseattrib_t ba
	 SET ba.BA_MCFID = NULL
	 WHERE ba.BA_MCFID = :OLD.MCF_ID;


END MCFEATURE_DEL_BEF_ROW_TRIG;

/
CREATE OR REPLACE TRIGGER MCFEATURE_INS_UPD_BEF_ROW_TRIG
BEFORE INSERT OR UPDATE OF MCF_CONID
ON MCFEATURE_T
FOR EACH ROW
DECLARE
	   v_conceptValid	BOOLEAN DEFAULT FALSE;
	   v_baseAttributeValid BOOLEAN DEFAULT FALSE;
	   v_numBaseAttr	NUMBER DEFAULT 0;
	   v_triggersOn         NUMBER DEFAULT NULL;


/******************************************************************************
   NAME:       MCFEATURE_INS_UPD_BEF_ROW_TRIG
   PURPOSE:    To maintain object consistency when a MultiColumnFeature
   			   is inserted or updated.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        25-02-2002  Olaf Rem         1. Created this trigger.
   1.1		  21-03-2002  Olaf Rem		   1. Adapted the function for the new
   			  			  	   			   	  BA_CONCEPT_T table.
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	  /* Check if the MultiColumnFeature has a Concept */
	  IF :NEW.MCF_CONID IS NULL
	  THEN
        v_conceptValid := FALSE;
	  ELSE
        v_conceptValid := TRUE;
	  END IF;


 	 /* Check if at least two BaseAttributes exist, which belong to the same
	    Concept as this MultiColumnFeature */
	 BEGIN
    	 SELECT COUNT(ba.ba_id)
    	 INTO 	v_numBaseAttr
    	 FROM 	BASEATTRIB_T   ba, BA_CONCEPT_T bc
    	 WHERE 	ba.BA_MCFID = :NEW.MCF_ID
		 AND    ba.BA_ID = bc.BC_BAID
    	 AND 	bc.BC_CONID = :NEW.MCF_CONID;

    	 IF (v_numBaseAttr >= 2)
    	 THEN
    	 	 v_baseAttributeValid := TRUE;
    	 ELSE
    	 	 v_baseAttributeValid := FALSE;
    	 END IF;

	 	 EXCEPTION
	     	WHEN NO_DATA_FOUND THEN
		 	v_baseAttributeValid := FALSE;
	 END;

	 /* Set validity flag */
	 IF (v_conceptValid  AND v_baseAttributeValid)
	 THEN
	 	 :NEW.MCF_VALID := 'YES';
	 ELSE
	 	 :NEW.MCF_VALID := 'NO';
     END IF;

END MCFEATURE_INS_UPD_BEF_ROW_TRIG;

/
CREATE OR REPLACE TRIGGER PARAMETER_I_U_D_AFT_STM_TRIG
 AFTER
 INSERT OR
 DELETE OR
 UPDATE OF PAR_OBJID, PAR_OBJTYPE, PAR_TYPE, PAR_STID
 ON PARAMETER_T
DECLARE
	   v_parId NUMBER DEFAULT NULL;
	   v_parNewStId NUMBER DEFAULT NULL;
	   v_parOldStId NUMBER DEFAULT NULL;
	   v_checkOldSt BOOLEAN DEFAULT FALSE;
	   v_triggersOn NUMBER  DEFAULT NULL;


/******************************************************************************
   NAME:       PARAMETER_I_U_D_AFT_STM_TRIG
   PURPOSE:    To manage object consistency when a Parameter is inserted
   			   or updated.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        7-03-2002   Olaf Rem         1. Created this trigger.
   1.1        11-07-2002  Erik Darwinkel   2. Added step deleted check.
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

  /* Quit, if the trigger-off-flag is set! */
  SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
  IF (v_triggersOn > 0)
  THEN
	RETURN;
  END IF;

	 -- only execute this if a Step is not being deleted!
  IF NOT (Glob_var_package.g_step_deleted)
  THEN

	 FOR i IN 1..Glob_var_package.t_par_id.COUNT LOOP
	 	 /*get the id of the step that corresponds to the Parameter that
		   was inserted or updated */
		 v_parNewStId := Glob_var_package.t_par_newstid(i);
		 v_parOldStId := Glob_var_package.t_par_oldstid(i);

		 IF (v_parNewStId <> v_parOldStId)
		 THEN
		 	 v_checkOldSt := TRUE;
		 END IF;

		 IF (v_parNewStId IS NOT NULL)
		 THEN
	 		 IF Is_step_valid(v_parNewStId)
			 THEN
			 	 UPDATE STEP_T st
				 SET st.ST_VALID = 'YES'
				 WHERE st.ST_ID = v_parNewStId;
			 ELSE
			 	 UPDATE STEP_T st
				 SET st.ST_VALID = 'NO'
				 WHERE st.ST_ID = v_parNewStId;
			 END IF;
		 END IF;

		 IF v_checkOldSt
		 THEN
			 IF (v_parOldStId IS NOT NULL)
			 THEN
		 		 IF Is_step_valid(v_parOldStId)
				 THEN
				 	 UPDATE STEP_T st
					 SET st.ST_VALID = 'YES'
					 WHERE st.ST_ID = v_parOldStId;
				 ELSE
				 	 UPDATE STEP_T st
					 SET st.ST_VALID = 'NO'
					 WHERE st.ST_ID = v_parOldStId;
				 END IF;
			 END IF;

		 END IF;

	 END LOOP;

	 /* reinitialize global variables */

	 Glob_var_package.g_par_counter := 0;
	 Glob_var_package.t_par_id.DELETE;
	 Glob_var_package.t_par_newstid.DELETE;
	 Glob_var_package.t_par_oldstid.DELETE;

  END IF;

END PARAMETER_I_U_D_AFT_STM_TRIG;

/
CREATE OR REPLACE TRIGGER PARAMETER_I_U_D_BEF_ROW_TRIG
 BEFORE 
 INSERT OR 
 DELETE OR 
 UPDATE OF PAR_OBJID, PAR_OBJTYPE, PAR_TYPE, PAR_STID 
 ON PARAMETER_T
FOR EACH ROW
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       PARAMETER_I_U_D_BEF_ROW_TRIG
   PURPOSE:    If a parameter gets inserted or updated the validity of the
   	 		   corresponding Step must be checked.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        07-03-2002  Olaf Rem         1. Created this trigger.
   1.1	      26-03-2002  "	   	   1. Adapted it to the M4 changes
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;

	 Glob_var_package.g_par_counter := Glob_var_package.g_par_counter + 1;
	 Glob_var_package.t_par_id(Glob_var_package.g_par_counter) := :NEW.PAR_ID;
	 Glob_var_package.t_par_newstid(Glob_var_package.g_par_counter) := :NEW.PAR_STID;
	 Glob_var_package.t_par_oldstid(Glob_var_package.g_par_counter) := :OLD.PAR_STID;

END PARAMETER_I_U_D_BEF_ROW_TRIG;

/
CREATE OR REPLACE TRIGGER RELATION_DEL_BEF_ROW_TRIG
 BEFORE DELETE ON RELATION_T
FOR EACH ROW
DECLARE

	  v_relId  	  	NUMBER	DEFAULT NULL;
	  v_triggersOn          NUMBER  DEFAULT NULL;

	  CURSOR c_inputParam IS
	  SELECT par.PAR_STID
	  FROM PARAMETER_T par
	  WHERE par.PAR_OBJID = :OLD.REL_ID
	  AND par.PAR_OBJTYPE = 'REL'
	  AND par.PAR_TYPE = 'IN';

	  CURSOR c_caseInput IS
	  SELECT cai.CAI_CAID, cai.CAI_OBJID
	  FROM CASEINPUT_T cai
	  WHERE cai.CAI_OBJID = :OLD.REL_ID
	  AND cai.CAI_OBJTYPE = 'REL';

	  CURSOR c_roleRes IS
	  SELECT rr.RR_RELID, rr.RR_FROMCONID
	  FROM ROLERESTRICTION_T rr
	  WHERE rr.RR_RELID = :OLD.REL_ID;

/******************************************************************************
   NAME:       RELATION_DEL_BEF_ROW_TRIG
   PURPOSE:    To manage object consistency when a relation is deleted

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        28-02-2002  Olaf Rem         1. Created this trigger.
   1.1        28-11-2002  Erik Darwinkel   1. Delete Coordinates.
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;

	 	  v_relId := :OLD.REL_ID;

	 /* DELETE documentation FOR this Relation */
	 Delete_documentation(:OLD.REL_ID, 'REL');

	 /* Delete ALL Coordinates, which USE this Relation */

	 DELETE HCI_COORD_T co
	 WHERE co.obj_id = v_relId;

	 /* DELETE ALL isA-relations FOR this Relation */
	 DELETE
	 FROM RELATIONISA_T risa
	 WHERE risa.RISA_SUPERRELID = :OLD.REL_ID
	 OR risa.RISA_SUBRELID = :OLD.REL_ID;

	 /* SET ALL Steps invalid, which USE this Relation AS input parameter */
	 BEGIN
    	 FOR r_par IN c_inputParam LOOP
    	 	 IF r_par.PAR_STID IS NOT NULL
			 THEN
    		 	 UPDATE STEP_T st
    			 SET st.ST_VALID = 'NO'
    			 WHERE st.ST_ID = r_par.PAR_STID;
    	 	 END IF;
    	 END LOOP;

		 EXCEPTION
		 	 WHEN NO_DATA_FOUND
			 THEN
			 NULL;
	 END;

	 /* SET CASE invalid, which uses this Relation AS caseInput */
	 FOR r_case IN c_caseInput LOOP
	 	 UPDATE CASE_T c
		 SET c.CA_VALID = 'NO'
		 WHERE c.CA_ID = r_case.CAI_CAID;

		 /* NOTE: row is deleted from the CASEINPUT_T table instead
		    of setting the object reference to NULL. M4 does NOT allow TO have
		    OBJID set to NULL for this table and why would you want to have caseinputs that
		    point to objects that have been deleted? */
		 DELETE
		 FROM CASEINPUT_T cai
		 WHERE cai.CAI_OBJID = r_case.CAI_OBJID
		 AND cai.CAI_OBJTYPE = 'REL';
	 END LOOP;

	 /* SET Concept invalid, which uses this Relation AS restrictionForRelation
	    within a RoleRestriction */
	 FOR r_role IN c_roleRes LOOP
	 	 UPDATE CONCEPT_T con
		 SET con.CON_VALID = 'NO'
		 WHERE con.CON_ID = r_role.RR_FROMCONID;

		 -- Set relation id to NULL within the rolerestriction table
		 UPDATE ROLERESTRICTION_T rr
		 SET rr.RR_RELID = NULL
		 WHERE rr.RR_RELID = r_role.RR_RELID;

	 END LOOP;


END RELATION_DEL_BEF_ROW_TRIG;

/
CREATE OR REPLACE TRIGGER RELATION_INS_UPD_BEF_TRIG
 BEFORE INSERT OR UPDATE OF REL_FROMCONID, REL_TOCONID, REL_FROMKID, REL_TOKID, REL_CSID
 ON RELATION_T
FOR EACH ROW
DECLARE
	   v_hasConcepts     BOOLEAN DEFAULT FALSE;
	   v_hasOneKeyHead   BOOLEAN DEFAULT FALSE;
	   v_hasTwoKeyHeads  BOOLEAN DEFAULT FALSE;
	   v_hasColumnSet    BOOLEAN DEFAULT FALSE;
	   v_triggersOn      NUMBER  DEFAULT NULL;

/******************************************************************************
   NAME:       RELATION_INS_UPD_BEF_TRIG
   PURPOSE:    set this Relation valid, if validity conditions are fulfilled, otherwise
   	    	   set this Relation invalid.

			   A Relation is valid, if
			    both included Concepts exist
			    at least one corresponding KeyHead (correspondsToForeignKey has a value)
		  	   exists
			    it has a corresponding ColumnSet (correspondsToColumnSet has a value)
		  	   if both KeyHeads have a value

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        28-02-2002  Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	  /* Check if both included Concepts exist
	     NOTE: it is not really checked here if the Concepts exist, but only
		 if there are references to two Concepts. */
	  IF (:NEW.REL_FROMCONID IS NOT NULL) AND (:NEW.REL_TOCONID IS NOT NULL)
	  THEN
	  	  v_hasConcepts := TRUE;
	  END IF;

	  /* Check if at least one corresponding KeyHead (correspondsToForeignKey has a value)
		  exists */
	  IF (:NEW.REL_FROMKID IS NOT NULL) OR (:NEW.REL_TOKID IS NOT NULL)
	  THEN
	  	  v_hasOneKeyHead := TRUE;
	  END IF;

	  IF (:NEW.REL_FROMKID IS NOT NULL) AND (:NEW.REL_TOKID IS NOT NULL)
	  THEN
	  	  v_hasTwoKeyHeads := TRUE;
	  END IF;

	  /* Check if it has a ColumnSet */
	  IF (:NEW.REL_CSID IS NOT NULL)
	  THEN
	  	  v_hasColumnSet := TRUE;
	  END IF;

	  /* Set the validity using the gathered information */

	  :NEW.REL_VALID := 'NO';

	  IF v_hasConcepts AND (NOT v_hasColumnSet) AND v_hasOneKeyHead
	  THEN
	  	  :NEW.REL_VALID := 'YES';
	  END IF;

	  IF v_hasConcepts AND v_hasColumnSet AND v_hasTwoKeyHeads
	  THEN
	  	  :NEW.REL_VALID := 'YES';
	  END IF;

END RELATION_INS_UPD_BEF_TRIG;

/
CREATE OR REPLACE TRIGGER ROLERESTR_I_U_AFT_STM_TRIG
 AFTER INSERT OR UPDATE OF RR_RELID, RR_FROMCONID, RR_TOCONID
 ON ROLERESTRICTION_T
DECLARE
	   v_conId NUMBER DEFAULT NULL;
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       ROLERESTR_I_U_AFT_STM_TRIG
   PURPOSE:    To manage object consistency when a RoleRestriction is inserted
   			   or updated.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        1-03-2002   Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

 	/* Retrieve from the t_rr_conid global PL/SQL table the Concepts
	   for which the RoleRestriction has changed and set their validity. */

	 FOR i IN 1..Glob_var_package.t_rr_conid.COUNT LOOP
	 	 /*get the id of the concept that corresponds to the RoleRestriction that
		   was inserted or updated */
		 v_conId := Glob_var_package.t_rr_conid(i);

		 IF Is_concept_valid(v_conId)
		 THEN
		 	 UPDATE CONCEPT_T con
			 SET con.CON_VALID = 'YES'
			 WHERE con.CON_ID = v_conId;
		 ELSE
		 	 UPDATE CONCEPT_T con
			 SET con.CON_VALID = 'NO'
			 WHERE con.CON_ID = v_conId;
		 END IF;

	 END LOOP;

	 /* reinitialize global variables */

	 Glob_var_package.g_rr_counter := 0;
	 Glob_var_package.t_rr_conid.DELETE;

END ROLERESTR_I_U_AFT_STM_TRIG;

/
CREATE OR REPLACE TRIGGER ROLERESTR_I_U_BEF_ROW_TRIG
 BEFORE INSERT OR UPDATE OF RR_RELID, RR_FROMCONID, RR_TOCONID 
 ON ROLERESTRICTION_T
FOR EACH ROW
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       ROLERESTR_I_U_BEF_ROW_TRIG
   PURPOSE:    If a role restriction gets inserted or updated the validity of the
   	 		   restrictionForConcept must be checked.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        01-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.
******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;


	 Glob_var_package.g_rr_counter := Glob_var_package.g_rr_counter + 1;
	 Glob_var_package.t_rr_conid(Glob_var_package.g_rr_counter) := :NEW.RR_FROMCONID;

	 IF (:NEW.RR_FROMCONID <> :OLD.RR_FROMCONID)
	 THEN
		 Glob_var_package.g_rr_counter := Glob_var_package.g_rr_counter + 1;
		 Glob_var_package.t_rr_conid(Glob_var_package.g_rr_counter) := :OLD.RR_FROMCONID;
	 END IF;

END ROLERESTR_I_U_BEF_ROW_TRIG;

/ 
CREATE OR REPLACE TRIGGER STEP_DEL_AFT_STM_TRIG
 AFTER DELETE ON STEP_T
DECLARE
	   v_stepId	 	   NUMBER DEFAULT NULL;
	   v_stepCaId 	   NUMBER DEFAULT NULL;
	   v_outputId	   NUMBER DEFAULT NULL;
	   v_outputType	   VARCHAR2(10)	  DEFAULT NULL;
	   v_outputParFound BOOLEAN DEFAULT FALSE;
	   v_numSteps	   NUMBER DEFAULT NULL;
	   v_triggersOn NUMBER DEFAULT NULL;

	   CURSOR c_inputPars IS
       SELECT par.PAR_STID
	   FROM PARAMETER_T par
	   WHERE par.PAR_TYPE = 'IN'
	   AND par.PAR_OBJID = v_outputId
	   AND par.PAR_OBJTYPE = v_outputType;

/******************************************************************************
   NAME:       STEP_DEL_AFT_STM_TRIG
   PURPOSE:    To perform work after a step is deleted.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        04-03-2002  Olaf Rem         1. Created this trigger.
   1.1		  26-03-2002  "	   			   1. Adapted it to M4 changes.
   1.2		  10-07-2002  Erik Darwinkel   1. Made fixes. 
   1.3        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

  /* Quit, if the trigger-off-flag is set! */
  SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
  IF (v_triggersOn > 0)
  THEN
	RETURN;
  END IF;

  /* DELETE documentation OF this Step */
	 FOR i IN 1..Glob_var_package.t_st_id.COUNT LOOP
	 	 v_stepId := Glob_var_package.t_st_id(i);

		 Delete_documentation(v_stepId, 'ST');
	 END LOOP;

  /* SET output parameter OF this Step invalid */
	 FOR i IN 1..Glob_var_package.t_st_id.COUNT LOOP
	 	 v_stepId := Glob_var_package.t_st_id(i);
		 v_outputId := Glob_var_package.t_st_p_objid(i);
		 v_outputType := Glob_var_package.t_st_p_objtype(i);
		 v_outputParFound := Glob_var_package.t_st_p_found(i);

		 -- depending on the type of the output parameter set the appropriate
		 -- object invalid: BaseAttribute or Concept
		 IF v_outputParFound
		 THEN
		 	 IF v_outputType = 'BA'
			 THEN
			 	 UPDATE BASEATTRIB_T ba
				 SET ba.BA_VALID = 'NO'
				 WHERE ba.BA_ID = v_outputId;
			 END IF;

			 IF v_outputType = 'CON'
			 THEN
			 	 UPDATE CONCEPT_T con
				 SET con.CON_VALID = 'NO'
				 WHERE con.CON_ID = v_outputId;
			 END IF;

  /* SET ALL Steps invalid, which USE the output parameter OF this Step AS input parameter */

		 -- find the steps that use the output parameter as input parameter
		 -- and set them invalid
		 	 FOR r_par IN c_inputPars LOOP
			 	 IF (r_par.PAR_STID IS NOT NULL)
				 THEN
				 	 UPDATE STEP_T st
					 SET st.ST_VALID = 'NO'
					 WHERE st.ST_ID = r_par.PAR_STID;
				 END IF;
			 END LOOP;

		 END IF;

	 END LOOP;


  /* SET CASE invalid, IF this Step IS the last existing step OF it */

	 FOR i IN 1..Glob_var_package.t_st_id.COUNT LOOP
	 	 v_stepCaId := Glob_var_package.t_st_caid(i);

		 BEGIN
		 	  SELECT COUNT(st.ST_ID)
			  INTO v_numSteps
			  FROM STEP_T st
			  WHERE st.ST_CAID = v_stepCaId;

			  IF (v_numSteps = 0)
			  THEN
				 UPDATE CASE_T ca
				 SET ca.CA_VALID = 'NO'
				 WHERE ca.CA_ID = v_stepCaId;
			  END IF;

		 	  EXCEPTION
			  	 WHEN NO_DATA_FOUND THEN
					  NULL;
		 END;

	 END LOOP;

	 -- set global variable (used by PARAMETER)
	 Glob_var_package.g_step_deleted := FALSE;
	 
END STEP_DEL_AFT_STM_TRIG;

/
CREATE OR REPLACE TRIGGER STEP_DEL_BEF_ROW_TRIG
 BEFORE DELETE ON STEP_T
FOR EACH ROW
DECLARE
	   v_stepId	  		NUMBER DEFAULT NULL;
	   v_stepCaId		NUMBER DEFAULT NULL;
	   v_outputId		NUMBER DEFAULT NULL;
	   v_outputType		VARCHAR2(10) DEFAULT NULL;
	   v_outputParFound	BOOLEAN DEFAULT FALSE;
	   v_triggersOn         NUMBER  DEFAULT NULL;

/******************************************************************************
   NAME:       STEP_DEL_BEF_ROW_TRIG
   PURPOSE:    To perform work as each row is deleted.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        04-03-2002  Olaf Rem         1. Created this trigger.
   1.1		  26-03-2002  "	   			   1. Adapted it to M4 changes.
   1.2        10-07-2002  Erik Darwinkel   1. Made fixes.
   1.3		  17-09-2002  "	   			   1. Adapted to M4 changes.
   1.4		  21-10-2002  "	   			   1. Added deletes in DBTRASH and M4TRASH
   1.5        06-02-2003  Martin Scholz    Trigger-off-flag tested.
******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* Store all id's of steps that are deleted.
	    Handle some delete actions in an after statement trigger. */
	  v_stepId := :OLD.ST_ID;
	  v_stepCaId := :OLD.ST_CAID;

	  Glob_var_package.g_st_counter := Glob_var_package.g_st_counter + 1;
	  Glob_var_package.t_st_id(Glob_var_package.g_st_counter) := v_stepId;
	  Glob_var_package.t_st_caid(Glob_var_package.g_st_counter) := v_stepCaId;

   /* DELETE ALL entries OF this Step in DBTrash table. */

	  DELETE
	  FROM  DBTRASH_T
	  WHERE STEPID = v_stepId;

   /* DELETE ALL entries OF this Step in M4Trash table. */

	  DELETE
	  FROM  M4TRASH_T
	  WHERE STEPID = v_stepId;

   /* DELETE ALL entries OF this Step in Stepsequence table. */

	  DELETE
	  FROM  STEPSEQUENCE_T sts
	  WHERE sts.STS_STID = v_stepId
	  OR    sts.STS_SUCCESSORSTID = v_stepId;

	  BEGIN
	 	  SELECT par.PAR_OBJID, par.PAR_OBJTYPE
		  INTO v_outputId, v_outputType
		  FROM PARAMETER_T par
		  WHERE par.PAR_STID = v_stepId
		  AND par.PAR_TYPE = 'OUT';

  	      v_outputParFound := TRUE;

		  EXCEPTION
		  	 WHEN NO_DATA_FOUND THEN
				  v_outputParFound := FALSE;
		  	 WHEN TOO_MANY_ROWS THEN
				  v_outputParFound := FALSE;

	  END;

	  Glob_var_package.t_st_p_objid(Glob_var_package.g_st_counter) := v_outputId;
	  Glob_var_package.t_st_p_objtype(Glob_var_package.g_st_counter) := v_outputType;
	  Glob_var_package.t_st_p_found(Glob_var_package.g_st_counter) := v_outputParFound;

  	 -- set global variable (used by PARAMETER)
	 Glob_var_package.g_step_deleted := TRUE;

  /* DELETE corresponding parameters OF this Step */

	 DELETE
	 FROM PARAMETER_T par
	 WHERE par.PAR_STID = v_stepId;

END STEP_DEL_BEF_ROW_TRIG;

/

CREATE OR REPLACE TRIGGER STEP_DEL_BEF_STM_TRIG
 BEFORE DELETE ON STEP_T
DECLARE
	   v_triggersOn NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       STEP_DEL_BEF_STM_TRIG
   PURPOSE:    To do initializations before a step is deleted.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        10-07-2002  Erik Darwinkel   1. Create trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.

******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;

	 /* initialize global variables */
	 Glob_var_package.g_st_counter := 0;
	 Glob_var_package.t_st_id.DELETE;
	 Glob_var_package.t_st_caid.DELETE;
	 Glob_var_package.t_st_p_objid.DELETE;
	 Glob_var_package.t_st_p_objtype.DELETE;
	 Glob_var_package.t_st_p_found.DELETE;

END STEP_DEL_BEF_STM_TRIG;

/
CREATE OR REPLACE TRIGGER STEP_INS_UPD_BEF_ROW_TRIG
 BEFORE INSERT OR UPDATE OF ST_CAID, ST_OPID
 ON STEP_T
FOR EACH ROW
DECLARE

	   v_hasCase  BOOLEAN DEFAULT FALSE;
	   v_hasOp	  BOOLEAN DEFAULT FALSE;
	   v_hasOutputPar  BOOLEAN DEFAULT FALSE;
	   v_hasInputPar   BOOLEAN DEFAULT FALSE;
	   v_hasValidOutput	  BOOLEAN DEFAULT FALSE;

	   v_stepId	  NUMBER DEFAULT NULL;
	   v_stepOpId NUMBER DEFAULT NULL;
	   v_stepCaId NUMBER DEFAULT NULL;
	   v_stepObjId NUMBER DEFAULT NULL;
	   v_stepObjType VARCHAR2(10) DEFAULT NULL;
	   v_numInputPars  NUMBER DEFAULT NULL;
   	   v_numOutputPars  NUMBER DEFAULT NULL;

	   v_triggersOn NUMBER DEFAULT NULL;
/******************************************************************************
   NAME:       STEP_INS_UPD_BEF_ROW
   PURPOSE:    To manage low level object consistency

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        05-03-2002  Olaf Rem         1. Created this trigger.
   1.1		  26-03-2002  "	   			   1. Adapted it to the M4 changes.
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.
******************************************************************************/
BEGIN

	/* Quit, if the trigger-off-flag is set! */
	SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	IF (v_triggersOn > 0)
    	THEN
		RETURN;
	END IF;


	 v_stepId := :NEW.ST_ID;
	 v_stepOpId := :NEW.ST_OPID;
	 v_stepCaId := :NEW.ST_CAID;

	 /* Store all id's of steps that are inserted or updated.
	    Handle some insert actions in an after statement trigger. */

  	 Glob_var_package.g_st_counter := Glob_var_package.g_st_counter + 1;
  	 Glob_var_package.t_st_id(Glob_var_package.g_st_counter) := v_stepId;
	 Glob_var_package.t_st_caid(Glob_var_package.g_st_counter) := v_stepCaId;

	 /*  SET this Step valid, IF validity conditions are fulfilled, otherwise
	      SET this Step invalid

		  A Step is valid, if
		   it has a Case (attribute belongsToCase has a value)
		   it has a corresponding Operator
		   at least one input and one output Parameter exists
		   the output parameter of this Step is not output of another Step */

	 -- Check if it has a Case
	 IF (v_stepCaId IS NOT NULL)
	 THEN
	 	 v_hasCase := TRUE;
	 END IF;

	 --Check if it has a corresponding Operator
	 IF (v_stepOpId IS NOT NULL)
	 THEN
	 	 v_hasOp := TRUE;
	 END IF;

	 -- Check if at least one input and one output Parameter exist
 	 BEGIN
	 	  SELECT par.PAR_OBJID, par.PAR_OBJTYPE
		  INTO v_stepObjId, v_stepObjType
		  FROM PARAMETER_T par
		  WHERE par.PAR_STID = v_stepId
		  AND par.PAR_TYPE = 'OUT';

  	      v_hasOutputPar := TRUE;

		  EXCEPTION
		  	 WHEN NO_DATA_FOUND THEN
				  v_hasOutputPar := FALSE;
		  	 WHEN TOO_MANY_ROWS THEN
				  v_hasOutputPar := FALSE; 	   -- more than one output. This invalidates the Step
	 END;

	 -- does at least one input exist?
 	 BEGIN
	 	  SELECT COUNT(par.PAR_ID)
		  INTO v_numInputPars
		  FROM PARAMETER_T par
		  WHERE par.PAR_STID = v_stepId
		  AND par.PAR_TYPE = 'IN';

 		  IF v_numInputPars > 0
		  THEN
		  	   v_hasInputPar := TRUE;
		  ELSE
		  	   v_hasInputPar := FALSE;
		  END IF;

		  EXCEPTION
		  	 WHEN NO_DATA_FOUND THEN
				  v_hasInputPar := FALSE;
	 END;

	 -- Check if the output parameter of this Step is not output of another Step
	 IF (v_hasOutputPar)
	 THEN
	 	 BEGIN
		 	 SELECT COUNT(par.PAR_ID)
			 INTO v_numOutputPars
			 FROM PARAMETER_T par
			 WHERE par.PAR_TYPE = 'OUT'
			 AND par.PAR_OBJID = v_stepObjId
			 AND par.par_OBJTYPE = v_stepObjType;

			 IF v_numOutputPars > 1
			 THEN
			 	  v_hasValidOutput := FALSE;
			 ELSE
			 	  v_hasValidOutput := TRUE;
			 END IF;

			 EXCEPTION
			 	  WHEN NO_DATA_FOUND THEN
				  	   v_hasValidOutput := FALSE;

		 END;
	 END IF;


	 -- Set Step validity
	 IF v_hasCase AND v_hasOp AND v_hasInputPar AND v_hasOutputPar AND v_hasValidOutput
	 THEN
	 	 :NEW.ST_VALID := 'YES';
	 ELSE
	 	 :NEW.ST_VALID := 'NO';
	 END IF;

	 /*  SET CASE valid, IF this Step IS the first step OF the corresponding CASE
	      (only  INSERT-TRIGGER)

		  Is handled in the after statement trigger */


	 /*  SET Concept valid, which IS output parameter OF this Step
 	     SET BaseAttribute valid, which IS output parameter OF this Step

		Is handled in the after statement trigger */


END STEP_INS_UPD_BEF_ROW_TRIG;

/
CREATE OR REPLACE TRIGGER STEP_I_U_AFT_STM_TRIG
 AFTER INSERT OR UPDATE OF ST_CAID, ST_OPID
 ON STEP_T
DECLARE

	   v_stepId	 NUMBER DEFAULT NULL;
	   v_stepCaId NUMBER DEFAULT NULL;
	   v_numSteps NUMBER DEFAULT NULL;
	   v_stepObjId NUMBER DEFAULT NULL;
	   v_stepObjType VARCHAR2(10) DEFAULT NULL;
	   v_hasOutputPar BOOLEAN DEFAULT FALSE;
	   v_triggersOn NUMBER DEFAULT NULL;
/******************************************************************************
   NAME:       STEP_I_U_AFT_STM_TRIG
   PURPOSE:    To manage low level object consistency

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        05-03-2002  Olaf Rem         1. Created this trigger.
   1.1        06-02-2003  Martin Scholz    Trigger-off-flag tested.
******************************************************************************/
BEGIN

	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;

	 /*  SET CASE valid, IF this Step IS the first step OF the corresponding CASE
	      (only  INSERT-TRIGGER). Uses Case-validity-function */
	 IF INSERTING
	 THEN
		 FOR i IN 1..Glob_var_package.t_st_id.COUNT LOOP
		 	 v_stepCaId := Glob_var_package.t_st_caid(i);
				  IF Is_case_valid(v_stepCaId)
				  THEN
				  	  UPDATE CASE_T ca
					  SET ca.CA_VALID = 'YES'
					  WHERE ca.CA_ID = v_stepCaId;
				  END IF;
		 END LOOP;
	 END IF;

	 /*  SET Concept valid, which IS output parameter OF this Step
 	     SET BaseAttribute valid, which IS output parameter OF this Step */

	 FOR i IN 1..Glob_var_package.t_st_id.COUNT LOOP
	 	 v_stepId := Glob_var_package.t_st_id(i);

	 	 BEGIN
		 	  SELECT par.PAR_OBJID, par.PAR_OBJTYPE
			  INTO v_stepObjId, v_stepObjType
			  FROM PARAMETER_T par
			  WHERE par.PAR_STID = v_stepId
			  AND par.PAR_TYPE = 'OUT';

	  	      v_hasOutputPar := TRUE;

			  EXCEPTION
			  	 WHEN NO_DATA_FOUND THEN
					  v_hasOutputPar := FALSE;
			  	 WHEN TOO_MANY_ROWS THEN	  -- more outputs found! Step is invalid. Validity of objects not set.
					  v_hasOutputPar := FALSE;
		 END;

		 IF v_hasOutputPar
		 THEN
		 	 IF v_stepObjType = 'CON'
			 THEN
			 	 IF Is_concept_valid(v_stepObjId)
				 THEN
				 	 UPDATE CONCEPT_T con
					 SET con.CON_VALID = 'YES'
					 WHERE con.CON_ID = v_stepObjId;
				 ELSE
				 	 UPDATE CONCEPT_T con
					 SET con.CON_VALID = 'NO'
					 WHERE con.CON_ID = v_stepObjId;
				 END IF;
			 END IF;

		 	 IF v_stepObjType = 'BA'
			 THEN
			 	 IF is_baseattribute_valid(v_stepObjId)
				 THEN
				 	 UPDATE BASEATTRIB_T ba
					 SET ba.BA_VALID = 'YES'
					 WHERE ba.BA_ID = v_stepObjId;
				 ELSE
				 	 UPDATE BASEATTRIB_T ba
					 SET ba.BA_VALID = 'NO'
					 WHERE ba.BA_ID = v_stepObjId;
				 END IF;
			 END IF;
		 END IF;

		 v_hasOutputPar := FALSE;
	 END LOOP;


	 /* reinitialize global variables */
	 Glob_var_package.g_st_counter := 0;
	 Glob_var_package.t_st_id.DELETE;
	 Glob_var_package.t_st_caid.DELETE;


END STEP_I_U_AFT_STM_TRIG;

/
CREATE OR REPLACE TRIGGER VALUE_DEL_BEF_ROW_TRIG
 BEFORE DELETE ON VALUE_T
FOR EACH ROW
DECLARE
	v_triggersOn NUMBER DEFAULT NULL;

	CURSOR c_steps IS
	  SELECT par.PAR_STID stid
	  FROM PARAMETER_T par
	  WHERE par.PAR_OBJID = :OLD.V_ID
	  AND par.PAR_OBJTYPE = 'V'
	  AND par.PAR_STID IS NOT NULL
	  ORDER BY par.PAR_STID;

	CURSOR c_cases IS
	  SELECT cai.CAI_CAID caid
	  FROM CASEINPUT_T cai
	  WHERE cai.CAI_OBJID = :OLD.V_ID
	  AND cai.CAI_OBJTYPE = 'V'
	  ORDER BY cai.CAI_CAID;
/******************************************************************************
   NAME:       VALUE_DEL_BEF_ROW_TRIG
   PURPOSE:    To manage object consistency when a value is deleted
   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        21-02-2002  Olaf Rem         1. Created this trigger.
   1.1        19-03-2002  Olaf Rem         1. Uses delete_documentation procedure
   			  			  	   			   2. Changes in setting steps invalid 
										      due to M4 changes.
   1.2        06-02-2003  Martin Scholz    Trigger-off-flag tested.
******************************************************************************/
BEGIN

	 /* Quit, if the trigger-off-flag is set! */
	 SELECT COUNT(*) INTO v_triggersOn FROM trigger_flag_t;
	 IF (v_triggersOn > 0)
    	 THEN
		RETURN;
	 END IF;

	 /* delete documentation of this Value */
	 Delete_documentation(:OLD.V_ID, 'V');

	 /* set all Steps invalid, which use this Value as input parameter. */

 	 FOR r_step IN c_steps LOOP
	 	 UPDATE STEP_T st
		 SET st.ST_VALID = 'NO'
   		 WHERE st.ST_ID = r_step.STID;
	 END LOOP;

	 /* set Case invalid, which uses this Value as caseInput */
	 FOR r_case IN c_cases LOOP
	 	 UPDATE CASE_T c
		 SET c.CA_VALID = 'NO'
		 WHERE c.CA_ID = r_case.CAID;
	 END LOOP;

END VALUE_DEL_BEF_ROW_TRIG;

/

commit;
quit
