CREATE OR REPLACE FUNCTION find_step_by_output_object(p_objid IN NUMBER, p_objtype IN VARCHAR2) 
RETURN NUMBER 
IS

  v_numSteps   NUMBER DEFAULT 0;
  v_stepId	   NUMBER DEFAULT -1; 

/******************************************************************************
   NAME:       FIND_STEP_BY_OUTPUT_OBJECT
   PURPOSE:    To check if an object(Concept or BaseAttribute) is output for a step
   			   and if so return the step id.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        28-03-2002  Olaf Rem        1. Created this function.

   PARAMETERS:
   INPUT:
   		 p_objid		  the object id
		 p_ojbtype		  the object type ('CON' for Concept, 'BA' for BaseAttribute)
   OUTPUT:
   		 v_stepId		  the id of the step where the object is output of (can be NULL!)
		 				  If no step can be found the value -1 is returned.
						  If more than one step is found the value -1 is returned.
   RETURNED VALUE:  NUMBER
   CALLED BY:
   CALLS:
   EXAMPLE USE:     v_stepId := find_step_by_output_object(v_conid, 'CON');
   ASSUMPTIONS:
   LIMITATIONS:
   ALGORITHM:
   NOTES:
******************************************************************************/
BEGIN

 	 BEGIN
   	 	 SELECT par.PAR_STID
   		 INTO v_stepId
   		 FROM PARAMETER_T par
   		 WHERE par.PAR_OBJID = p_objid
   		 AND par.PAR_OBJTYPE = p_objtype
		 AND par.PAR_TYPE = 'OUT';

		 RETURN v_stepId;

   		 EXCEPTION
   	     WHEN NO_DATA_FOUND THEN
   		 	  RETURN -1;
   		 
   	     WHEN TOO_MANY_ROWS THEN
   		 	  RETURN -1;

	 END;

END find_step_by_output_object;

/ 
CREATE OR REPLACE FUNCTION Is_baseattribute_valid(p_baid IN NUMBER)
RETURN BOOLEAN
IS
  	   v_baType	     VARCHAR2(15) DEFAULT NULL;
	   v_hasConcept	 BOOLEAN   DEFAULT FALSE;
	   v_numCols     NUMBER    DEFAULT NULL;
	   v_hasColumn	 BOOLEAN   DEFAULT FALSE;
	   v_hasStep	 BOOLEAN   DEFAULT FALSE;
	   v_paramStId	 NUMBER    DEFAULT NULL;
	   v_numSteps	 NUMBER    DEFAULT NULL;
	   v_numConcepts NUMBER    DEFAULT NULL;

/******************************************************************************
   NAME:       IS_BASEATTRIBUTE_VALID
   PURPOSE:    To check if the given BaseAttribute is valid or not

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        26-03-2002  Olaf Rem        1. Created this function.
   1.1        19-04-2002  Erik Darwinkel  2. Changes for new table BA_COLUMN_T 
                                        
  
   PARAMETERS:
   INPUT:	  		p_baid
   OUTPUT:  		TRUE or FALSE depending if the object is valid or not
   RETURNED VALUE:  BOOLEAN
   CALLED BY:
   CALLS:
   EXAMPLE USE:     if IS_BASEATTRIBUTE_VALID(ba_id) then ...
   ASSUMPTIONS:
   LIMITATIONS:
   ALGORITHM:
   	 A BaseAttribute is valid if:
	 ? it has a Concept (belongsToConcept has a value)
	 ? it has a corresponding Column (correspondsToColumn has a value), for type DB
	 ? a Step exists, where this BaseAttribute is an output parameter, for type MINING

   NOTES:
******************************************************************************/
BEGIN

	 /* SET this BaseAttribute valid, IF validity conditions are fulfilled,
	 otherwise SET this BaseAttribute invalid (inserts AND updates) */

	 --does it have a concept?
  	  BEGIN
	  	   SELECT COUNT(bc.BC_CONID)
		   INTO v_numConcepts
		   FROM BA_CONCEPT_T bc
		   WHERE bc.BC_BAID = p_baid;

		   IF (v_numConcepts > 0)
		   THEN
		       v_hasConcept := TRUE;
		   ELSE
		       RETURN FALSE;
		   END IF;

		   EXCEPTION
		       WHEN NO_DATA_FOUND
			   THEN
			       RETURN FALSE;
	  END;

	 --does it have a column if of type DB?
	 BEGIN
	 	  SELECT COUNT(*)
		  INTO   v_numCols
		  FROM   BASEATTRIB_T ba
		  ,      BA_COLUMN_T bc
		  WHERE  ba.BA_ID = p_baid
		  AND    ba.BA_ATTRIBTYPE = 'DB'
		  AND    ba.BA_ID = bc.BAC_BAID
		  ;

 		  IF (v_numCols > 0)
		  THEN
		 	 v_hasColumn := TRUE;
		  END IF;

    	  EXCEPTION
		     WHEN NO_DATA_FOUND THEN
			      RETURN FALSE;

	 END;


	 --does a Step exist where it is an output parameter if of type MINING?
	 
 	 BEGIN
	 	  SELECT ba.BA_ATTRIBTYPE
		  INTO   v_baType
		  FROM   BASEATTRIB_T ba
		  WHERE  ba.BA_ID = p_baid;

    	  EXCEPTION
		     WHEN NO_DATA_FOUND THEN
			      RETURN FALSE;

	 END;

	 IF (v_baType = 'MINING')
	 THEN
	 	 v_hasStep := Is_output_for_a_step(p_baid, 'BA');
		 IF (NOT v_hasStep) OR (v_hasStep IS NULL)
		 THEN
		 	 RETURN FALSE;
		 END IF;
	 END IF;

	 IF v_hasConcept AND (v_hasColumn OR v_hasStep)
	 THEN
	 	 RETURN TRUE;
	 ELSE
	 	 RETURN FALSE;
     END IF;

END Is_baseattribute_valid;

/
CREATE OR REPLACE FUNCTION Is_case_valid (p_caid IN NUMBER) 
RETURN BOOLEAN 
IS 

	   v_hasStep  		 	 BOOLEAN DEFAULT FALSE;
	   v_hasValidCaInput	 BOOLEAN DEFAULT FALSE;
	   v_hasValidTargetAttr  BOOLEAN DEFAULT FALSE;
	   v_hasCaseOutput   	 BOOLEAN DEFAULT FALSE;
	   v_hasValidPopulation	 BOOLEAN DEFAULT FALSE;

	   v_caseId	  	  NUMBER DEFAULT NULL;
	   v_casePopId    NUMBER DEFAULT NULL;
	   v_caseOutput   NUMBER DEFAULT NULL;
	   v_casePopulation NUMBER DEFAULT NULL;

	   v_dummyValue	  NUMBER DEFAULT NULL;

	   CURSOR c_caInputs IS
	   SELECT cai.CAI_OBJID, cai.CAI_OBJTYPE
	   FROM CASEINPUT_T cai
	   WHERE cai.CAI_CAID = p_caid;

	   CURSOR c_caTargets IS
	   SELECT caa.CAA_OBJID, caa.CAA_OBJTYPE
	   FROM CASEATTRIB_T caa
	   WHERE caa.CAA_CAID = p_caid;

/******************************************************************************
   NAME:       IS_CASE_VALID
   PURPOSE:    To check if a Case is valid given its id.

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        07-03-2002  Olaf Rem         1. Created this function.

   PARAMETERS:
   INPUT:	  p_caid	  The id of the case that should be checked
   OUTPUT:	  A boolean indicating if the case is valid or not
   RETURNED VALUE:  BOOLEAN
   CALLED BY:
   CALLS:
   EXAMPLE USE:      IF (IS_CASE_VALID(v_caId) THEN ...
   ASSUMPTIONS:
   LIMITATIONS:
   ALGORITHM:
   			 A Case is valid, if
			  at least one Step exists
			  all objects of caseInput, if defined, must exist (object-references 
			   must exist for all existing entries)
			  all BaseAttributes for targetAttributes, if defined, must exist
			  a Concept for caseOutput exists
			  a population exists which is member of caseInput

   NOTES:
******************************************************************************/
BEGIN


	 /* Check if AT LEAST one Step EXISTS */
	 BEGIN
	 	  SELECT COUNT(st.ST_ID)
		  INTO v_dummyValue
		  FROM STEP_T st
		  WHERE st.ST_CAID = p_caid;

		  IF (v_dummyValue > 0)
		  THEN
		  	  v_hasStep := TRUE;
		  ELSE
		  	  RETURN FALSE;
		  END IF;

		  EXCEPTION
		      WHEN NO_DATA_FOUND THEN
			      RETURN NULL;
	 END;

	 /* Check if ALL objects OF caseInput, IF defined, exist (object-references
	    must exist FOR ALL existing entries) */
     v_hasValidCaInput := TRUE;
 	 FOR r_caInput IN c_caInputs LOOP
		 IF (r_caInput.CAI_OBJID IS NULL) OR (r_caInput.CAI_OBJTYPE IS NULL)
		 THEN
			 RETURN FALSE;
		 END IF;
	 END LOOP;

	 /* Check if ALL BaseAttributes FOR targetAttributes, IF defined, exist */
		v_hasValidTargetAttr := TRUE;
		FOR r_caTarget IN c_caTargets LOOP
			IF (r_caTarget.CAA_OBJID IS NULL) OR (r_caTarget.CAA_OBJTYPE IS NULL)
			THEN
				RETURN FALSE;
			END IF;
		END LOOP;

	 /* Check if a Concept FOR caseOutput EXISTS */
	 
	 BEGIN
	 	  SELECT ca.CA_OUTPUT
		  INTO v_caseOutput
		  FROM CASE_T ca
		  WHERE ca.CA_ID = p_caid;

		  IF (v_caseOutput IS NOT NULL)
		  THEN
		  	  v_hasCaseOutput := TRUE;
		  ELSE
		  	  RETURN FALSE;
		  END IF;

		  EXCEPTION
		      WHEN NO_DATA_FOUND THEN
			  	   RETURN NULL;
	 END;

	 /* Check if a population EXISTS which IS member OF caseInput */

	 BEGIN
	 	  SELECT ca.CA_POPULATION
		  INTO v_casePopulation
		  FROM CASE_T ca
		  WHERE ca.CA_ID = p_caid;

		  IF (v_casePopulation IS NOT NULL)
		  THEN
		  	  v_hasValidPopulation := TRUE;
		  ELSE
		  	  RETURN FALSE;
		  END IF;

		  EXCEPTION
		      WHEN NO_DATA_FOUND THEN
			  	   RETURN NULL;
	 END;

	 --check if this population is member of CaseInput
	 IF (v_hasValidPopulation)
	 THEN
	 	 BEGIN
		 	  SELECT COUNT(cai.CAI_ID)
			  INTO v_dummyValue
			  FROM CASEINPUT_T cai
			  WHERE cai.CAI_OBJID = v_casePopulation
			  AND cai.CAI_OBJTYPE = 'CON';

			  IF v_dummyValue = 1
			  THEN
			  	  v_hasValidPopulation := TRUE;
			  ELSE
			  	  RETURN FALSE;
			  END IF;

			  EXCEPTION
			      WHEN NO_DATA_FOUND THEN
				 	 v_hasValidPopulation := TRUE;
		 END;
	 END IF;

	 /* Return the Case validity using the gathered information */
	 IF v_hasStep AND v_hasValidCaInput	AND v_hasValidTargetAttr AND v_hasCaseOutput AND v_hasValidPopulation
	 THEN
	 	 RETURN TRUE;
	 ELSE
	     RETURN FALSE;
	 END IF;

END Is_case_valid;

/ 
CREATE OR REPLACE FUNCTION Is_concept_valid (p_conid IN NUMBER)
RETURN BOOLEAN
IS
   v_numColSets	  NUMBER		 DEFAULT NULL;
   v_hasColumnSet BOOLEAN		 DEFAULT FALSE;
   v_hasStep 	  BOOLEAN		 DEFAULT FALSE;
   v_hasBaseAttribute 	  BOOLEAN		 DEFAULT FALSE;
   v_hasRoleRes   BOOLEAN		 DEFAULT FALSE;
   v_roleResValid BOOLEAN		 DEFAULT FALSE;
   v_conType	  VARCHAR2(10)	 DEFAULT NULL;
   v_conRoleResid NUMBER		 DEFAULT NULL;
   v_opId		  NUMBER		 DEFAULT NULL;
   v_stId		  NUMBER		 DEFAULT NULL;
   v_stDummy	  NUMBER		 DEFAULT NULL;
   v_numAttr	  NUMBER		 DEFAULT NULL;

   CURSOR c_roleRes IS
   SELECT rr.RR_FROMCONID, rr.RR_TOCONID, rr.RR_RELID
   FROM ROLERESTRICTION_T rr
   WHERE rr.RR_FROMCONID = p_conid;

/******************************************************************************
   NAME:       IS_CONCEPT_VALID
   PURPOSE:    Check if a Concept is valid

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        28-02-2002  Olaf Rem         1. Created this function.
   1.1		  25-03-2002  Olaf Rem		   1. Adapted to the M4 changes
   1.2		  02-05-2002  Erik Darwinkel   1. Adapted to the M4 changes

   PARAMETERS:
   INPUT:
   		  p_conid		  The id of the Concept
   OUTPUT: Boolean indicating if the Concept is valid or not
   RETURNED VALUE:  BOOLEAN
   CALLED BY:		BA_CONCEPT_I_U_AFT_STM_TRIG
   CALLS:
   EXAMPLE USE:     IF IS_CONCEPT_VALID(v_conid) THEN ...
   ASSUMPTIONS:
   LIMITATIONS:
   ALGORITHM:
   			 A Concept is valid, if
			 ? it has a corresponding ColumnSet (correspondsToColumnSet has a value), for type DB
			 ? a Step exists, where this Concept is an output parameter, for type MINING
			 ? at least one included BaseAttribute exists
			 ? if RoleRestrictions exist for this Concept (attribute restrictionForConcept),
			 then they must have a value for restrictionToConcept and restrictionForRelation
   NOTES:
******************************************************************************/

BEGIN
	 /* Get the concept type if available */
	 BEGIN
	 	  SELECT con.CON_TYPE
		  INTO   v_conType
		  FROM   CONCEPT_T con
		  WHERE  con.CON_ID = p_conid;

		  EXCEPTION
		  	  WHEN NO_DATA_FOUND THEN
			  	   RETURN FALSE;		 --when concept cannot be found, validity is false
	 END;

	 /* Get the number of concept columnsets */
	 BEGIN
	 	  SELECT COUNT(*)
		  INTO   v_numColSets
		  FROM   columnset_t con
		  WHERE  con.cs_conid = p_conid;

	 END;

	 /* Check if the concept type is NULL */
	 IF v_conType IS NULL
	 THEN
	 	 RETURN NULL;	 			 --if Concept type is not known, validity is not known
	 END IF;

	 /* Check if the Concept has one or more ColumnSets if the Concept is of type DB */
	 IF (v_conType = 'DB') AND (v_numColSets > 0)
	 THEN
	 	 v_hasColumnSet := TRUE;
	 ELSE
	 	 v_hasColumnSet := FALSE;
	 END IF;

	 /* Check if a Step exists, where this Concept is an output parameter, for type MINING */
	 IF v_conType = 'MINING'
	 THEN
	 	 v_hasStep := Is_output_for_a_step(p_conid, 'CON');
		 IF (NOT v_hasStep) OR (v_hasStep IS NULL)
		 THEN
		 	 RETURN FALSE;
		 END IF;
	 END IF;

 	 /* Check if at least one included BaseAttribute exists */
	 BEGIN
    	 SELECT COUNT(bc.BC_BAID)
    	 INTO 	v_numAttr
    	 FROM 	BA_CONCEPT_T  bc
    	 WHERE 	bc.BC_CONID = p_conid;

		 IF v_numAttr > 0
		 THEN
		 	 v_hasBaseAttribute := TRUE;
		 ELSE
		 	 RETURN FALSE;		   		-- if no BaseAttribute exists the Concept is invalid
		 END IF;

	 EXCEPTION
	     WHEN NO_DATA_FOUND THEN
		 	  RETURN FALSE;			   	-- if no BaseAttribute exists the Concept is invalid
	 END;

	 /* If RoleRestrictions exist for this Concept (attribute restrictionForConcept),
		then they must have a value for restrictionToConcept and restrictionForRelation */
	 v_roleResValid := TRUE;
	 FOR r_role IN c_roleRes LOOP
	 	 v_hasRoleRes := TRUE;
		 IF (r_role.RR_TOCONID IS NULL) OR (r_role.RR_RELID IS NULL)
		 THEN
		 	 v_roleResValid := FALSE;
		 END IF;
	 END LOOP;

	 /* Now set the validity of the Concept using the information gathered previously */
	 IF (v_hasColumnSet OR v_hasStep) AND v_hasBaseAttribute AND (NOT v_hasRoleRes)
	 THEN
	 	 RETURN TRUE;
	 END IF;

	 IF (v_hasColumnSet OR v_hasStep) AND v_hasBaseAttribute AND v_hasRoleRes AND v_roleResValid
	 THEN
	 	 RETURN TRUE;
	 END IF;

	 RETURN FALSE;

END Is_concept_valid;

/
CREATE OR REPLACE FUNCTION Is_mcfeature_valid (p_mcfid IN NUMBER)
RETURN BOOLEAN
IS
   v_numBaseAttr  NUMBER   DEFAULT 0;
   v_mcfConId	  NUMBER   DEFAULT NULL;   
/******************************************************************************
   NAME:       IS_MCFEATURE_VALID
   PURPOSE:    Check if a MultiColumnFeature is valid

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        22-02-2002  Olaf Rem         1. Created this function.
   1.1        21-03-2002  Olaf Rem		   1. Adapted the function to the new
   			  			  	   			      BA_CONCEPT_T table.

   PARAMETERS:
   INPUT:
   		  p_mcfid		  The id of the MultiColumnFeature

   OUTPUT:
   		   Boolean indicating if the MultiColumnFeature is valid or not
   RETURNED VALUE:  BOOLEAN
   CALLED BY:		BASEATTRIB_I_AFT_STM_TRIG
   CALLS:
   EXAMPLE USE:     IF IS_MCFEATURE_VALID(v_mcfid) THEN ...
   ASSUMPTIONS:
   LIMITATIONS:
   ALGORITHM:
   			   A MultiColumnFeature is valid, if
			     it has a Concept (belongsToConcept has a value)
				 at least two BaseAttributes exist, which belong
				  to the same Concept than this MultiColumnFeature
   NOTES:
******************************************************************************/
BEGIN
	 /* Check if the MultiColumnFeature has a Concept */
	 BEGIN
	 	  SELECT mcf.MCF_CONID
		  INTO v_mcfConId
		  FROM MCFEATURE_T mcf
		  WHERE mcf.MCF_ID = p_mcfid;
		  
		  EXCEPTION
		  	  WHEN NO_DATA_FOUND THEN
			  	   RETURN FALSE; 	 		  --when no mcf is found return false	 	 	
	 END;
	 
	 /* If no Concept exists the MultiColumnFeature is invalid */
	 IF v_mcfConId IS NULL
	 THEN
	 	 RETURN FALSE;
	 END IF;

 	 /* Check if at least two BaseAttributes exist, which belong to the same
	    Concept as this MultiColumnFeature */
	 BEGIN
    	 SELECT COUNT(ba.ba_id)
    	 INTO 	v_numBaseAttr
    	 FROM 	BASEATTRIB_T   ba, BA_CONCEPT_T bc
    	 WHERE 	ba.BA_MCFID = p_mcfid
		 AND    ba.BA_ID = bc.BC_BAID
    	 AND 	bc.BC_CONID = v_mcfConId;

    	 IF (v_numBaseAttr >= 2)
    	 THEN
    	 	 RETURN TRUE;
    	 ELSE
    	 	 RETURN FALSE;
    	 END IF;

	 EXCEPTION
	     WHEN NO_DATA_FOUND THEN
		 	  RETURN FALSE;
	 END;

END Is_mcfeature_valid;

/
CREATE OR REPLACE FUNCTION Is_output_for_a_step(p_objid IN NUMBER, p_objtype IN VARCHAR2)
RETURN BOOLEAN
IS

  v_numSteps   NUMBER DEFAULT 0;

/******************************************************************************
   NAME:       IS_OUTPUT_FOR_A_STEP
   PURPOSE:    To check if an object(Concept or BaseAttribute) is output for a step

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        28-03-2002  Olaf Rem        1. Created this function.
   1.1        05-12-2002  Erik Darwinkel  1. Added concept output check.

   PARAMETERS:
   INPUT:
   		 p_objid		  the object id
		 p_ojbtype		  the object type ('CON' for Concept, 'BA' for BaseAttribute)
   OUTPUT:
   		 TRUE, FALSE or NULL
		 	   		 	  Returns TRUE if the object is output for a step.
						  Returns FALSE if the object is not output for a step.
						  Returns NULL if the object is output for more than one step.

   RETURNED VALUE:  BOOLEAN
   CALLED BY:
   CALLS:
   EXAMPLE USE:     if v_conid IS_OUTPUT_FOR_A_STEP(v_conid, 'CON') then ...
   ASSUMPTIONS:
   LIMITATIONS:
   ALGORITHM:
   NOTES:
******************************************************************************/
BEGIN

 	 BEGIN
   	 	 SELECT COUNT(par.PAR_STID)
   		 INTO v_numSteps
   		 FROM PARAMETER_T par
   		 WHERE par.PAR_OBJID = p_objid
   		 AND par.PAR_OBJTYPE = p_objtype
		 AND par.PAR_TYPE = 'OUT';

		 IF (v_numSteps = 0)
		 THEN
		 	 IF (p_objtype = 'BA')
		   	 THEN /* Look if concept is output. */
		   	 	 SELECT COUNT(par.PAR_STID)
				 INTO v_numSteps
				 FROM	ba_concept_t bc
		 	 	 ,		parameter_t par
   		 	 	 WHERE	bc.BC_BAID = p_objid
		 	 	 AND	bc.BC_CONID =  par.PAR_OBJID
   		 	 	 AND	par.PAR_OBJTYPE = 'CON'
		 	 	 AND	par.PAR_TYPE = 'OUT';
		 		 IF (v_numSteps = 1)
		 		 THEN
		 	 	 	 RETURN TRUE;
				 ELSE
		 	 	 	 RETURN FALSE;
		 		 END IF;
			 ELSE
		 	 	 RETURN FALSE;
			 END IF;
		 END IF;

		 IF (v_numSteps = 1)
		 THEN
		 	 RETURN TRUE;
		 END IF;

		 IF (v_numSteps > 1)
		 THEN
		 	 RETURN NULL;
		 END IF;

   		 EXCEPTION
   	     WHEN NO_DATA_FOUND THEN
   		 	  RETURN FALSE;
	 END;

END Is_output_for_a_step;

/
CREATE OR REPLACE FUNCTION Is_step_valid (p_stid IN NUMBER)
RETURN BOOLEAN
IS
    v_hasCase  	  	  BOOLEAN DEFAULT FALSE;
    v_hasOp			  BOOLEAN DEFAULT FALSE;
    v_hasInputPar	  BOOLEAN DEFAULT FALSE;
    v_hasOutputPar	  BOOLEAN DEFAULT FALSE;
    v_hasValidOutput  BOOLEAN DEFAULT FALSE;

	v_stepOpId		  NUMBER DEFAULT NULL;
	v_stepCaId		  NUMBER DEFAULT NULL;
	v_stepObjId		  NUMBER DEFAULT NULL;
	v_stepObjType	  VARCHAR2(20) DEFAULT NULL;
	v_numInputPars	  NUMBER DEFAULT NULL;
	v_numOutputPars   NUMBER DEFAULT NULL;

/******************************************************************************
   NAME:       IS_STEP_VALID
   PURPOSE:    To check if a step is valid given its id

   REVISIONS:
   Ver        Date        Author           Description
   ---------  ----------  ---------------  ------------------------------------
   1.0        07-03-2002  Olaf Rem         1. Created this function.
   1.1		  26-03-2002  Olaf Rem		   1. Adapted it to fit the M4 changes

   PARAMETERS:
   INPUT:	  p_stid	  The id of the step that should be checked
   OUTPUT:	  		Boolean indicating if the input step is valid or not
   RETURNED VALUE:  BOOLEAN
   CALLED BY:
   CALLS:
   EXAMPLE USE:     IF (IS_STEP_VALID(v_stepId)) THEN...
   ASSUMPTIONS:
   LIMITATIONS:
   ALGORITHM:  		A Step is valid, if
				    it has a Case (attribute belongsToCase has a value)
				    it has a corresponding Operator
				    at least one input and one output Parameter exists
				    the output parameter of this Step is not output of
				     another Step
   NOTES:

******************************************************************************/
BEGIN

	 -- Check if it has a Case
	 BEGIN
	 	  SELECT st.ST_CAID
		  INTO v_stepCaId
		  FROM STEP_T st
		  WHERE st.ST_ID = p_stid;

		  IF (v_stepCaId IS NOT NULL)
		  THEN
		  	  v_hasCase := TRUE;
		  ELSE
		  	  RETURN FALSE;
		  END IF;

		  EXCEPTION
		      WHEN NO_DATA_FOUND THEN
			  	   RETURN NULL;
	 END;


	 --Check if it has a corresponding Operator
 	 BEGIN
	 	  SELECT st.ST_OPID
		  INTO v_stepOpId
		  FROM STEP_T st
		  WHERE st.ST_ID = p_stid;

		  IF (v_stepOpId IS NOT NULL)
		  THEN
		  	  v_hasOp := TRUE;
		  ELSE
		  	  RETURN FALSE;
		  END IF;

		  EXCEPTION
		      WHEN NO_DATA_FOUND THEN
			  	   RETURN NULL;
	 END;

	 -- Check if at least one input and one output Parameter exist
 	 BEGIN
	 	  SELECT par.PAR_OBJID, par.PAR_OBJTYPE
		  INTO v_stepObjId, v_stepObjType
		  FROM PARAMETER_T par
		  WHERE par.PAR_STID = p_stid
		  AND par.PAR_TYPE = 'OUT';

 		  v_hasOutputPar := TRUE;

		  EXCEPTION
		  	 WHEN NO_DATA_FOUND THEN
				  v_hasOutputPar := FALSE;
		     WHEN TOO_MANY_ROWS THEN
		 	  	  RETURN FALSE;
	 END;

	 -- can an input parameter be found?
 	 BEGIN
	 	  SELECT COUNT(par.PAR_ID)
		  INTO v_numInputPars
		  FROM PARAMETER_T par
		  WHERE par.PAR_STID = p_stid
		  AND par.PAR_TYPE = 'IN';

	  	  IF v_numInputPars > 0
		  THEN
		  	   v_hasInputPar := TRUE;
		  ELSE
		  	   v_hasInputPar := FALSE;
		  END IF;

		  EXCEPTION
		  	 WHEN NO_DATA_FOUND THEN
			 	  v_hasInputPar := FALSE;
	 END;

	 -- Check if the output parameter of this Step is not output of another Step
	 IF (v_hasOutputPar)
	 THEN
	 	 BEGIN
		 	 SELECT COUNT(par.PAR_ID)
			 INTO v_numOutputPars
			 FROM PARAMETER_T par
			 WHERE par.PAR_TYPE = 'OUT'
			 AND par.PAR_OBJID = v_stepObjId
			 AND par.par_OBJTYPE = v_stepObjType;

			 IF v_numOutputPars > 1
			 THEN
			 	  v_hasValidOutput := FALSE;
			 ELSE
			 	  v_hasValidOutput := TRUE;
			 END IF;

			 EXCEPTION
			 	  WHEN NO_DATA_FOUND THEN
				  	   v_hasValidOutput := FALSE;

		 END;
	 END IF;


	 -- Return Step validity
	 IF v_hasCase AND v_hasOp AND v_hasInputPar AND v_hasOutputPar AND v_hasValidOutput
	 THEN
	 	 RETURN TRUE;
	 ELSE
	 	 RETURN FALSE;
	 END IF;

END Is_step_valid;

/

commit;
quit
