package Bachelorpackage;

/*
 * Copyright 2017 Andreas Sitta
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *  http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;

/** Diese Klasse stelt Routinen zur Verfügung um eine Ausgabe von Kanten und Knoten im gephi-Format ( *.gexf) zu erstellen. Die Datei basiert auf dem XML Format.
*  Verwendung in der folgenden Reihenfolge.
*  1. open erstellte eine neue leere Datei. Der Parameter AF bestimmt, ob im Kopf zusätzliche Attribute erstellt werden. 
*  2. writeNodes speichert einen Knoten.
*  3. writeEdge speichert eine Kante.
*  4. close schreibt den Kopf, die Knoten und Kanten in die Datei und schließt dies korrekt ab.
*/

public class gexfExport {
	static BufferedWriter rGXF;
	static ArrayList<String> Nodes, Edges;
	
	/** open erstellt eine leere Datei
	* @param filename 			Name der Datei einschließlich Pfad
	*/
	public static void open(String fileName, boolean apf)  throws IOException  {
		rGXF = new BufferedWriter(new FileWriter(fileName+".gexf"));
		Nodes = new ArrayList<String>();
		Edges = new ArrayList<String>();
	}
	
	/** write Header schreibt den gephi conformen xml Header an den Anfang der Datei, wird von close aufgerufen.
	*/
	public static void writeHeader()  throws IOException  {
		rGXF.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?> \r\n");
		rGXF.write("<gexf xmlns=\"http://www.gexf.net/1.2draft\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.gexf.net/1.2draft http://www.gexf.net/1.2draft/gexf.xsd\" version=\"1.2\"> \r\n");
		rGXF.write("    <meta lastmodifieddate=\"2009-03-20\"> \r\n");
		rGXF.write("        <creator>Gephi.org</creator> \r\n");
		rGXF.write("        <description>A Web network</description> \r\n");
		rGXF.write("    </meta> \r\n");
		rGXF.write("<graph>\r\n");
	  	rGXF.write("   <attributes class=\"node\"> \r\n");
	   	rGXF.write("   <attribute id=\"0\" title=\"community\" type=\"integer\"> \r\n");
	   	rGXF.write("       <default>0</default> \r\n");
	   	rGXF.write("   </attribute> \r\n");
	   	rGXF.write("</attributes> \r\n");
	}

	/** writeNode(int id, int source, String label) schreibt die Daten eines Knotens in einen Puffer
	* @param id		 			fortlaufende Nummer des Knoten
	* @param source		 	ursprüngliche id des Knoten - todo übergabe als zusätzliches Attribut
	* @param label 			Bezeichnung des Knotens
	*/
	public static boolean writeNode(int id, int source, String label)  throws IOException  {
		if (label =="") 	Nodes.add("<node id=\""+id+"\" />  \r\n");
		else					Nodes.add("<node id=\""+id+"\" label=\""+label+" />  \r\n");
		return  true;
	}

	/** writeNode(int id, int source, String labe, int community) schreibt die Daten eines Knotens in einen Puffer
	* @param id 				fortlaufende Nummer des Knoten
	* @param source 			ursprüngliche id des Knoten - todo übergabe als zusätzliches Attribut
	* @param label 			Bezeichnung des Knotens
	* @param community 	Zugeordnete Community
	*/
	public static boolean writeNode(int id, int source, String label, int community)  throws IOException  {
		// Ungültige Zeichen im Namen löschen
		label=label.replaceAll("[^A-Za-z]", "");

		if (label =="") 	Nodes.add("<node id=\""+id+"\"  >   \r\n");
		else 					Nodes.add("<node id=\""+id+"\" label=\""+label+"\" >   \r\n");
		Nodes.add("<attvalues>  \r\n");
		Nodes.add("<attvalue for=\"0\" value=\""+community+"\" />  \r\n");
		Nodes.add("</attvalues>  \r\n");
		Nodes.add("</node>\r\n");
		return  true;
	}

	/** writeEdge(int id, int source, int target, boolean dir, double weight) schreibt die Daten einer Kante in einen Puffer
	* @param id 				fortlaufende Nummer der Kante
	* @param source 			id des source Knotens
	* @param target 			id des target Knotens
	* @param dir				Richtung der Kante true=directed / false = undirected
	* @param weight	 		Gewicht der Kante
	*/
	public static boolean writeEdge(int id, int source, int target, boolean dir, double weight)  throws IOException  {
		String s  = "<edge id=\""+id+" \" source=\""+source+"\" target=\""+target+"\" type=\"";
		
		if (dir = true)  s += "directed\"";
		else  s +="undirected\"";
		
		if (weight <= 0) s +=" />  \r\n";
		else  s +=" weight=\""+weight+"\" />  \r\n";

		Edges.add(s);
		return  true;
	}

	/** writeEdge(int id, int source, int target, boolean dir, double weight) schreibt die Daten einer Kante in einen Puffer
	* @param id 					fortlaufende Nummer der Kante
	* @param source 			id des source Knotens
	* @param target 			id des target Knotens
	* @param dir		 			Richtung der Kante true=directed / false = undirected
	*/
	public static boolean writeEdge(int id, int source, int target, boolean dir)  throws IOException  {
		String s ="<edge id=\""+id+" \" source=\""+source+"\" target=\""+target+"\" type=\"";
	
		if (dir = true)  s += "directed\"  /> \r\n";
		else  s +="undirected\"  /> \r\n";
		
		Edges.add(s);
		return  true;
	}

	/** writeEdge(int id, int source, int target, boolean dir, double weight) schreibt die Daten einer "directed" Kante in einen Puffer
	* @param id 				fortlaufende Nummer der Kante
	* @param source 			id des source Knotens
	* @param target 			id des target Knotens
	*/
	public static boolean writeEdge(int id, int source, int target)  throws IOException  {
		Edges.add("<edge id=\""+id+" \" source=\""+source+"\" target=\""+target+"\" type=\"directed\"  /> \r\n");
		return  true;
	}

	/** close 	
	 *  1. Schreibt Header,
	 *  2. Schreibt Knoten, wenn vorhanden,
	 *  3. Schreibt Kanten, wenn vorhanden,
	 *  4. Schließt xml ab und schließt Datei.
	*/
	public static boolean close()  throws IOException  {
		writeHeader();
		
		if (Nodes.size() > 0)  {
			rGXF.write("<nodes>\r\n");
		    for (int i = 0; i <Nodes.size(); i++) {
				rGXF.write(Nodes.get(i));
		    }
		    rGXF.write("</nodes>\r\n");
		}    

		if (Edges.size() > 0)  {
			rGXF.write("<edges>\r\n");
			for (int i = 0; i <Edges.size(); i++) {
				rGXF.write(Edges.get(i));
			}
			rGXF.write("</edges> \r\n");
		}
		
		rGXF.write("</graph> \r\n");
		rGXF.write("</gexf> \r\n");
		rGXF.close();
		return  true;
	}

}
