package edu.udo.cs.yale.operator.exercise.fpGrowth;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

public class FPTreeNode {

	protected int frequency = 0;
	protected Item nodeItem;
	protected FPTreeNode sibling;
	protected FPTreeNode father;
	protected Map<Item,FPTreeNode> children;
public FPTreeNode() {
	children = new HashMap<Item, FPTreeNode>();
}
public FPTreeNode(FPTreeNode father, Item nodeItem) {
	this.father = father;
	children = new HashMap<Item, FPTreeNode>();
	this.nodeItem = nodeItem;
}

/**
 * This method adds a set of Items to the tree of this node. This set of items 
 * has to be sorted after the frequency of the contained items. This method is 
 * recursivly used to expand the tree for the given set, by adding a node for 
 * the first item and then call this method with the remaining set on the new 
 * node. The frequency of the set is represented of weight. siblingChain is 
 * the headerTable, giving this method a startingpoint for finding the other nodes 
 * of the item to append new nodes
 * @param itemSet the sorted set of items
 * @param weight the frequency of the set of items
 * @param siblingChain gives the headertable for finding other nodes of an item
 */
public void addItemSet(Collection<Item> itemSet, int weight, Map<Item, FPTreeNode> siblingChain){
	Iterator<Item> iterator = itemSet.iterator();
	if (iterator.hasNext()) {
		Item currentItem = iterator.next();
		FPTreeNode childNode;
		if(!children.containsKey(currentItem)) {
			childNode = createChildNode(currentItem);
			children.put(currentItem, childNode);
			// append new children to sibling chain
			if (!siblingChain.containsKey(currentItem)) {
				siblingChain.put(currentItem, childNode);
			} else {
				FPTreeNode currentNode = siblingChain.get(currentItem);
				while (currentNode.hasSibling()) {
					currentNode = currentNode.getSibling();
				}
				currentNode.setSibling(childNode);
			}
		} else {
			childNode = children.get(currentItem);
		}		
		itemSet.remove(currentItem);
		childNode = children.get(currentItem); 
		childNode.increaseFrequency(weight);
		childNode.addItemSet(itemSet, weight, siblingChain);
	}
}
/**
 * This method prunes the tree recursivly by testing if the current node
 * is less frequent than the minimal needed value given by minSupport and
 * numberOfTransactions 
 * @param minSupport is the minimal Support needed to be a frequent set
 * @param numberOfTransactions is the total number of transactions of the exampleset
 */
public void prune(double minSupport, int numberOfTransactions){
	Collection<Item> items = children.keySet();
	Collection<Item> removeItems = new ArrayList<Item>();
	Iterator<Item> iterator = items.iterator();
	int minFrequency = (int)(((double) numberOfTransactions) * minSupport);
	while(iterator.hasNext()) {
		Item currentItem = iterator.next();
		FPTreeNode currentNode = children.get(currentItem);
		if (currentNode.getFrequency() < minFrequency) {
			//no deletion allowed while iterating: Queue for delayed deletion
			removeItems.add(currentItem);
		} else {
			currentNode.prune(minSupport, numberOfTransactions);
		}
	}
	//delayed deletion
	iterator = removeItems.iterator();
	while (iterator.hasNext()) {
		children.remove(iterator.next());
	}
}
/**
 * Returns the father of this node or null if node is root
 */
public FPTreeNode getFather() {
	return father;
}
/**
 * Returns true if node has father. If node is root, false is returned
 */
public boolean hasFather() {
	return (this.father!=null);
}
/**
 * Returns the next node representing the same item as this node.
 */
public FPTreeNode getSibling() {
	return sibling;
}
/**
 * Returns the last node of the chain of nodes representing the same item as this node
 */
public FPTreeNode getLastSibling() {
	FPTreeNode currentNode = this;
	while (currentNode.hasSibling()) {
		currentNode = currentNode.getSibling();
	}
	return currentNode;
}
/**
 * This method sets the next node in the chain of node representing 
 * the same item as this node
 * @param sibling is the next node in the chain
 */
public void setSibling(FPTreeNode sibling) {
	this.sibling = sibling;
}
/**
 * Returns true if this node is not the last one in the chain of nodes representing 
 * the same item as this node. Otherwise false is returned.
 */
public boolean hasSibling() {
	return (this.sibling!=null);
}
/**
 * This method increases the frequency of this node by the given weight
 * @param weight the frequency is increased by weight
 */
public void increaseFrequency(int weight) {
	this.frequency = this.frequency + weight;
}
/**
 * this returns the frequency of the node
 */
public int getFrequency() {
	return this.frequency;
}
/**
 * this returns the item, this node represents
 */
public Item getNodeItem() {
	return this.nodeItem;
}
/**
 * This returns the map, which maps the child nodes on items. It may be used 
 * to get a set of all childNodes or all represented items. 
 * @return
 */
public Map<Item,FPTreeNode> getChildren() {
	return this.children;
}
/**
 * This method returns the first child. If no child exists, null is returned
 */
public FPTreeNode getChild(){
	if (children.size() != 1 ) {
		return null;
	} else {
		return children.get(children.keySet().iterator().next());
	}
}
/**
 * this method creates a new childnode of this node, representing the node item
 * @param nodeItem the item, represented by the new node
 * @return
 */
public FPTreeNode createChildNode(Item nodeItem){
	return new FPTreeNode(this, nodeItem);
}
}
