package edu.udo.cs.yale.operator.exercise.fpGrowth;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import edu.udo.cs.yale.operator.UserError;

public abstract class FPTree extends FPTreeNode {
	protected double minSupport;
	protected int numberOfTransactions;
	protected FrequentItemSets rules;
	protected Map<Item,FPTreeNode> siblingChain;
	protected Collection<Item> conditionalItems;
	private Class treeMinerClass;
	public FPTree() {
		super();
		siblingChain = new HashMap<Item,FPTreeNode>();
		children = new HashMap<Item,FPTreeNode>();		
		conditionalItems = new LinkedList<Item>();
	}
	/**
	 * This method adds a set of Items to the tree. This set of items has to be sorted
	 * after the frequency of the contained items. This method should be used to add
	 * Items of a transaction or a treepath to the tree. The frequency of the set is
	 * represented of weight, which should be 1 if items are gathered from transaction
	 * @param itemSet the sorted set of items
	 * @param weight the frequency of the set of items
	 */
	public void addItemSet(Collection<Item> itemSet, int weight){
		addItemSet(itemSet, weight, siblingChain);
	}
	/**
	 * This method should be implemented to mine the tree. All found frequent sets
	 * can be added to rules for returning them.
	 * @param frequentSets is the container for found frequent sets
	 * @throws UserError if the treeMinerClass cannot be instanciated
	 */
	public abstract void mineTree(FrequentItemSets frequentSets) throws UserError; 
	/**
	 * isSinglePath returns true if the given tree consists only of a 
	 * single path, otherwise returns false.
	 * @param tree is the tree to be tested
	 * @return true if tree is single path, false otherwise
	 */
	protected boolean isSinglePath(FPTree tree) {
		if (children.size() > 1 ) {
			return false;
		} else {
			FPTreeNode currentNode = children.get(children.keySet().iterator().next());
			while(currentNode.getChildren().size() == 1) {
				currentNode = currentNode.getChild();
			}
			if (currentNode.getChildren().size() == 0 ) {
				return true;
			} else {
				return false;
			}
		}
	}
	/**
	 * getPrefixOfNode returns a Collection of Items, which are the fathers of the 
	 * given node in descending order given by the tree. This ensures that the collection is sorted
	 * by the frequency of the nodes. 
	 * @param node is the node, which fathers are returned
	 */
	protected Collection<Item> getPrefixOfNode(FPTreeNode node) {
		List<Item> prefix = new LinkedList<Item>();
		FPTreeNode currentNode = node;
		while (currentNode.hasFather()) {
			currentNode = currentNode.getFather();
			prefix.add(currentNode.getNodeItem());
		}
		//List is in wrong order, reverse!
		Collections.reverse(prefix);
		return prefix;
	}
	/**
	 * This method returns a new empty conditional tree for recursiv mining. 
	 * @param condition is a Collection of Items representing the condition of the tree 
	 * @throws UserError
	 */
	public FPTree getEmptyConditionalTree(Collection<Item> condition) throws UserError {
		try {
			FPTree tree = (FPTree) treeMinerClass.newInstance();
			// setting parameters (needed because of creation with reflection)
			tree.setTreeClass(treeMinerClass);
			tree.setConditionalItems(condition);
			tree.setMinimalSupport(minSupport);
			tree.setNumberOfTransactions(numberOfTransactions);
			tree.setRulesContainer(rules);
			return tree;			
		} catch (InstantiationException exception) {
			throw new UserError(null, 904, treeMinerClass.getName(), exception.getMessage());
		} catch (IllegalAccessException exception) {
			throw new UserError(null, 904, treeMinerClass.getName(), exception.getMessage());
		}
	}
	// Setters needed to construct proper FPTree with reflection
	public FPTreeNode createChildNode(Item nodeItem){
		return new FPTreeNode(null, nodeItem);
	}
	public void setMinimalSupport(double minSupport) {
		this.minSupport = minSupport;
	}
	public void setNumberOfTransactions(int numberOfTransactions) {
		this.numberOfTransactions = numberOfTransactions;		
	}
	public void setRulesContainer(FrequentItemSets rules) {
		this.rules = rules;
	}
	public void setConditionalItems(Collection<Item> conditionalItems) {
		this.conditionalItems = conditionalItems;
	}
	public void setTreeClass(Class treeMinerClass) {
		this.treeMinerClass = treeMinerClass;
	}
}
