package edu.udo.cs.yale.operator.exercise.fpGrowth;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import edu.udo.cs.yale.example.Attribute;
import edu.udo.cs.yale.example.Example;
import edu.udo.cs.yale.example.ExampleSet;
import edu.udo.cs.yale.operator.IOObject;
import edu.udo.cs.yale.operator.Operator;
import edu.udo.cs.yale.operator.OperatorDescription;
import edu.udo.cs.yale.operator.OperatorException;
import edu.udo.cs.yale.operator.UserError;
import edu.udo.cs.yale.operator.parameter.ParameterType;
import edu.udo.cs.yale.operator.parameter.ParameterTypeDouble;
import edu.udo.cs.yale.operator.parameter.ParameterTypeString;
import edu.udo.cs.yale.tools.Tools;

public class FPGrowth extends Operator {
	private double minSupport;
	public FPGrowth(OperatorDescription description){
		super(description);
	}
	
	public IOObject[] apply() throws OperatorException {
		String treeMiner = getParameterAsString("name_of_implementing_class");
		try {
			Class treeMinerClass = Tools.classForName(treeMiner);
			ExampleSet exampleSet = (ExampleSet) getInput(ExampleSet.class);
			FrequentItemSets rules = new FrequentItemSets();
			minSupport = getParameterAsDouble("min_support");
			int minFrequency = (int) ((double)minSupport * exampleSet.size());
			// precomputing data properties
			int numberOfAttributes = exampleSet.getNumberOfAttributes();
			boolean[] useAttributes = isBooleanAttribute(exampleSet);
			Map<Attribute,Item> mapping = getAttributeMapping(exampleSet, useAttributes);
			// computing frequency of 1-Item Sets and eliminating non frequent items
			getFrequentItems(exampleSet, mapping, useAttributes);
			for (int i = 0; i < numberOfAttributes; i++) {
				useAttributes[i] = useAttributes[i] && mapping.get(exampleSet.getAttribute(i)).getFrequency() >= minFrequency;
			}
			
			// generating FP Tree
			FPTree tree = getFPTree(exampleSet, mapping, useAttributes, rules, treeMinerClass);
			tree.mineTree(rules);

			return new IOObject[] {rules};
		} catch (ClassNotFoundException exception) {
			throw new UserError(null, 904, treeMiner, exception.getMessage());
		}

	}
	/**
	 * Returns a new FPTree, representing the complete ExampleSet.
	 * @param exampleSet is the exampleSet, which shall be represented
	 * @param mapping is the mapping of attributes of the exampleSet to items
	 * @param useAttributes is an boolean array, true if attribute shall be used
	 * @param rules is the container for the later mined frequent item sets
	 * @param treeMinerClass is the class implementing the mining method
	 * @throws UserError if treeMinerClass cannot be created
	 */
	@SuppressWarnings("unchecked")
	private FPTree getFPTree(ExampleSet exampleSet, Map<Attribute, Item> mapping, boolean[] useAttributes, FrequentItemSets rules, Class treeMinerClass) throws UserError {
		try {
			FPTree tree = (FPTree) treeMinerClass.newInstance();
			// setting parameters (needed because of creation with reflection)
			tree.setTreeClass(treeMinerClass);
			tree.setMinimalSupport(minSupport);
			tree.setNumberOfTransactions(exampleSet.size());
			tree.setRulesContainer(rules);
			int numberOfAttributes = exampleSet.getNumberOfAttributes();
			Iterator<Example> iterator = exampleSet.iterator();
			while (iterator.hasNext()) {
				Example currentExample = iterator.next();
				List<Item> itemSet = new ArrayList<Item>();
				for (int i = 0; i < numberOfAttributes; i++) {
					Attribute currentAttribute = exampleSet.getAttribute(i);
					if (useAttributes[i] && currentExample.getValue(currentAttribute) == exampleSet.getAttribute(i).mapString("true")) {
						itemSet.add(mapping.get(currentAttribute));
					}
				}
				Collections.sort(itemSet);
				tree.addItemSet(itemSet, 1);
			}
			return tree;			
		} catch (InstantiationException exception) {
			throw new UserError(null, 904, treeMinerClass.getName(), exception.getMessage());
		} catch (IllegalAccessException exception) {
			throw new UserError(null, 904, treeMinerClass.getName(), exception.getMessage());
		}

	}
	/**
	 * returns an boolean array, if an attribute has boolean values
	 * @param exampleSet exampleSet, which attributes are tested
	 */
	private boolean[] isBooleanAttribute(ExampleSet exampleSet) {
		//computing Attributes to test, because only boolean attributes are used
		int numberofAttributes = exampleSet.getNumberOfAttributes();
		boolean[] useAttributes = new boolean[numberofAttributes];
		for (int i = 0; i < numberofAttributes; i++) {
			useAttributes[i] = exampleSet.getAttribute(i).isBooleanClassification();
		}
		return useAttributes;		
	}
	/**
	 * this method maps the attributes of the given exampleSet, which should be used 
	 * to an Item. If an attribute should be used is given by the boolean array 
	 * useAttribues
	 * @param exampleSet the exampleSet which attributes are mapped
	 * @param useAttributes the boolean array representing the use of an attribute
	 * @return
	 */
	private Map<Attribute,Item> getAttributeMapping(ExampleSet exampleSet, boolean[] useAttributes) {
		//computing Attributes to test, because only boolean attributes are used
		int numberOfAttributes = useAttributes.length;
		Map<Attribute,Item> mapping = new HashMap<Attribute,Item>();
		Attribute[] attributes = new Attribute[numberOfAttributes];
		for (int i = 0; i < numberOfAttributes; i++) {
			attributes[i] = exampleSet.getAttribute(i);
			if (useAttributes[i]) {
				mapping.put(attributes[i], new Item(attributes[i].getName()));
			}
		}
		return mapping;
	}
	/**
	 * This method scans the exampleSet and counts the frequency of every item
	 * @param exampleSet the exampleSet to be scaned
	 * @param mapping the mapping of attributes to items
	 * @param useAttributes the boolean array representing if an attribute should be used
	 */
	private void getFrequentItems(ExampleSet exampleSet, Map<Attribute,Item> mapping, boolean[] useAttributes) {
		int numberOfAttributes = exampleSet.getNumberOfAttributes();
		//iterate over exampleSet, counting item frequency
		Iterator<Example> iterator = exampleSet.iterator();
		while (iterator.hasNext()) {
			Example currentExample = iterator.next();
			for (int i = 0; i < numberOfAttributes; i++) {
				//if attribute is boolean and if attribute is true increase frequency of item
				if (useAttributes[i] &&	currentExample.getValue(exampleSet.getAttribute(i)) == exampleSet.getAttribute(i).mapString("true")) {
					mapping.get(exampleSet.getAttribute(i)).increaseFrequency();
				}
			}
		}
	}
	

	public Class[] getInputClasses() {
		return new Class[] { ExampleSet.class };
	}

	public Class[] getOutputClasses() {
		return new Class[] { FrequentItemSets.class };
	}
	public List<ParameterType> getParameterTypes() {
		List<ParameterType> types = super.getParameterTypes();
		ParameterType type = new ParameterTypeString("name_of_implementing_class", "Fully qualified classname of the class implementing FPGrowthApplier", "FPGrowthApplier");
		type.setExpert(false);
		types.add(type);
		type = new ParameterTypeDouble("min_support", "Minimal Support", 0, 1, 0.5);
		type.setExpert(false);
		types.add(type);
		return types;
	}
}
